// TryShutdownPriv.c: Attempts to enable SE_SHUTDOWN_NAME privilege and
//                    call ExitWindowsEx.

// Written by Shawn Fessenden 11/99, Oswego Software, Inc.
// 630-554-3567       Voice
// 630-554-0200       Fax
// www.oswegosw.com   Web
// shawn@oswegosw.com Mail

// This is an example published on the VEE Reflector mailing list and is
// donated to the public domain by the author.

// Call Shutdown to shutdown Windows. The passed parameter is passed through
// ExitWindowsEx with bit 4 masked off. Bit 4 (0x10) is used by this function
// to determine whether or not to try to adjust privileges. If bit 4 is 1,
// then this function will attempt to adjust privileges if the call to
// ExitWindowsEx fails. If bit 4 is 0, it will not.
// This function returns whatever ExitWindowsEx returns.

// The ExitWindowsEx flags are:
//   EWX_LOGOFF   0
//   EWX_SHUTDOWN 1
//   EWX_REBOOT   2
//   EWX_FORCE    4
//   EWX_POWEROFF 8


#define WIN32_LEAN_AND_MEAN
#include <windows.h>

#include <malloc.h>

// Local functions.
BOOL SeeIfWeCanShutdown(UINT uFlags);
BOOL EnableShutdownPrivilege();


__declspec(dllexport) BOOL Shutdown(UINT uFlags)
{
	BOOL	bRes;

	// First, call ExitWindowsEx.
	if(!(bRes = SeeIfWeCanShutdown(uFlags)) && (uFlags & 0x10)) {

		// See if we can enable SE_SHUTDOWN_NAME & try again.
		EnableShutdownPrivilege();
		return SeeIfWeCanShutdown(uFlags);
	}

	// If we get here then the user doesn't want to try again.
	return bRes;
}

BOOL SeeIfWeCanShutdown(UINT uFlags)
{return ExitWindowsEx((uFlags & ~0x10), 0);}

BOOL EnableShutdownPrivilege()
{
	// Normal stuff.
	BOOL		bRet;
	BOOL		bFound;
	DWORD		i, dwSizeRequired;
	PVOID		pvBuf;
	HANDLE	hToken;
	
	// Special stuff.
	LUID							luidShutdown;
	PTOKEN_PRIVILEGES	ptp;

	// Inititialize some stuff.
	bRet		= FALSE;	//Assume failure.
	bFound	= FALSE;	//Assume we can't find SE_SHUTDOWN_NAME.
	pvBuf		= NULL;		//Token privileges buffer.

	// Get access token. Fail if we can't open for at least query access. Well, try adjust too.
	if(!OpenProcessToken(GetCurrentProcess(), TOKEN_QUERY | TOKEN_ADJUST_PRIVILEGES, &hToken)) return FALSE;

	// Get required buffer size for token privileges & allocate buffer.
	GetTokenInformation(hToken, TokenPrivileges, NULL, 0, &dwSizeRequired);
	if(pvBuf = malloc(dwSizeRequired)) {

		// Get token information.
		if(GetTokenInformation(hToken, TokenPrivileges, pvBuf, dwSizeRequired, &dwSizeRequired)) {

			// Get LUID for shutdown privilege.
			if(LookupPrivilegeValue(NULL, SE_SHUTDOWN_NAME, &luidShutdown)) {

				// Iterate the list of privileges and enable shutdown if found.
				ptp = (PTOKEN_PRIVILEGES)pvBuf;
				for(i = 0; i < ptp->PrivilegeCount; i++) {
					if(memcmp(&ptp->Privileges[i].Luid, &luidShutdown, sizeof(LUID)) == 0) {
						ptp->Privileges[i].Attributes |= SE_PRIVILEGE_ENABLED;
						bFound = TRUE;
						break;
					}
				}

				// If we found it try to adjust privileges.
				if(bFound) {
					if(AdjustTokenPrivileges(hToken, FALSE, ptp, dwSizeRequired, NULL, &dwSizeRequired)) bRet = TRUE;
				}

				// Clean up as we fall out.
			}
		}
		free(pvBuf);
	}
	CloseHandle(hToken);

	return bRet;
}
