/////////////////////////////////////////////////////////////////////////////
// VEEWINAPI.CPP: Handy stuff for VEE 5.01.

/////////////////////////////////////////////////////////////////////////////
// Windows Common.
#define STRICT
#define WIN32_LEAN_AND_MEAN
#include <windows.h>

#include <math.h>
#include <tchar.h>
#include <stdio.h>

/////////////////////////////////////////////////////////////////////////////
// Structure used for child window enumeration.
typedef struct _WINNODE {	//WindowNode, wn: Used to keep track of VEE child windows.
	struct _WINNODE *pNext;	//Pointer to next in chain.
	HWND						hWnd;		//Window handle.
	PTCHAR					pText;	//Ptr to window text.
} WINNODE, *PWINNODE;

/////////////////////////////////////////////////////////////////////////////
// GLOBALS.
static HWND				ghWnd		= NULL;					//Set at process attach time.
static RECT				grcWnd	= {0, 0, 0, 0};	//Just because it's handy.
static DWORD			gdwErr	= 0;						//Also because it's handy.
static HINSTANCE	ghInst	= NULL;					//Not used here, but standard.

static TCHAR			gszInitFailureTitle[]	= _T("VeeWinapi Initialization Failure");
static TCHAR			gszInitFailureMsg[]		= _T("Could not find VEE window.");

// For our subclassing example.
static HWND				ghWndButton			= NULL;	//This window is subclassed.
static WNDPROC		gpfnPrevWndProc	= NULL;	//This is the WNDPROC we replaced.
static HFONT			ghButtonFont		= NULL;	//Button font.
static PTCHAR			gpButtonText		=	NULL;	//Button text.
static COLORREF		gcrTextColor		= 0;		//Button text color.
static COLORREF		gcrBkgndColor		= 0;		//Button background color.

static PWINNODE		gpWinRoot				= NULL;	//Root of WINNODE chain.

static TCHAR			gszUnsubclsFailTitle[]	= _T("Un-subclassing Failed");
static TCHAR			gszUnsubclsFailMsg[]		= _T("Sorry kids, but VEE is gonna crash");

LRESULT CALLBACK	MyWndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam);

/////////////////////////////////////////////////////////////////////////////
// Helper Prototypes.
BOOL							ChangeStyleBit(LONG lStyleBit, BOOL bNewState);
HWND							FindVeeChildWindow(PTCHAR pText);
BOOL CALLBACK			EnumChildWindowProc(HWND hWnd, LPARAM lParam);
PWINNODE					AllocWinNode();
VOID							DeleteWinNodes(PWINNODE pNode);

/////////////////////////////////////////////////////////////////////////////
// Export Macro.
#define DllExport __declspec(dllexport)

/////////////////////////////////////////////////////////////////////////////
// Export Prototypes.
extern "C" {
	DllExport BOOL	ShowSysMenu(BOOL bShow);
	DllExport BOOL	ShowMinBox(BOOL bShow);
	DllExport BOOL	ShowMaxBox(BOOL bShow);
	DllExport UINT	GetSysDir(PTCHAR pDir);
	DllExport UINT	GetWinDir(PTCHAR pDir);
	DllExport DWORD	GetCurDir(PTCHAR pDir);
	DllExport BOOL	SetCurDir(PTCHAR pDir);
	DllExport VOID	GetSysInfo(PSHORT psProcArch, PINT piPageSize, PINT piProcMask,
										PINT piNumProcs, PINT piProcType, PINT piAllocGran,
										PSHORT psProcLevel, PSHORT psProcRev);
	DllExport INT		GetBootType(VOID);
	DllExport BOOL	GetScreenSize(PINT piSizeX, PINT piSizeY);
	DllExport BOOL	GetWorkareaRect(PINT piLeft, PINT piTop, PINT piRight,
																PINT piBottom);
	DllExport BOOL	IsSlowMachine();
	DllExport BOOL	GetLoggedOnUserName(PTCHAR pName);
	DllExport BOOL	GetExtendedVersionInfo(PINT piMajor, PINT piMinor,
									 PINT piBuild, PINT piPlatform, PTCHAR pSvcPack);
	DllExport DWORD	GetLastWinErr();
	DllExport BOOL	GetLastWinErrMsg(DWORD dwErrCode, PTCHAR pErrMsg);
	DllExport	VOID	SetLastWinErr(INT iErr);
	DllExport BOOL	SubclassButton(PTCHAR pText);
	DllExport BOOL	SetButtonText(PTCHAR pText);
	DllExport BOOL	SetButtonFont(INT iSize, PTCHAR pName);
	DllExport VOID	SetButtonBkgndColor(INT r, INT g, INT b);
	DllExport VOID	SetButtonTextColor(INT r, INT g, INT b);
	DllExport VOID	RedrawButton();
	DllExport VOID	ClickButton();
	DllExport BOOL	UnsubclassButton();
	DllExport BOOL	VeeBeep(UINT uType);
	DllExport BOOL	ControlledBeep(DWORD dwFreq, DWORD dwDur);
}

/////////////////////////////////////////////////////////////////////////////
// Entry point.
BOOL WINAPI DllMain(HINSTANCE hInst, DWORD dwReason, LPVOID lpvReserved)
{
	switch(dwReason) {

		// When attaching, find the VEE main window.
		case DLL_PROCESS_ATTACH:
			ghWnd = FindWindow(_T("HPVEE"), NULL);
			if(ghWnd) ghInst = hInst;

			// If we can't find the window, fail.
			else {
				MessageBox(NULL, gszInitFailureMsg, gszInitFailureTitle, MB_OK | MB_ICONEXCLAMATION);
				return FALSE;
			}
		break;

		// Note that we're not doing any per-thread initialization.
		case DLL_THREAD_ATTACH: break;
		case DLL_THREAD_DETACH: break;

		// When we unload, make *sure* to un-subclass our button.
		case DLL_PROCESS_DETACH:
			if(gpfnPrevWndProc) UnsubclassButton();
			ghInst = NULL;
		break;

		// If Windows passed anything else, aliens have taken over.
		default: return FALSE;
	}

	// Tell Windows we loaded successfully.
	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// Exported functions.
BOOL ShowSysMenu(BOOL bShow)
{return ChangeStyleBit(WS_SYSMENU, bShow);}

BOOL ShowMinBox(BOOL bShow)
{return ChangeStyleBit(WS_MINIMIZEBOX, bShow);}

BOOL ShowMaxBox(BOOL bShow)
{return ChangeStyleBit(WS_MAXIMIZEBOX, bShow);}

UINT GetSysDir(PTCHAR pDir)
{return GetSystemDirectory(pDir, lstrlen(pDir));}

UINT GetWinDir(PTCHAR pDir)
{return GetWindowsDirectory(pDir, lstrlen(pDir));}

DWORD GetCurDir(PTCHAR pDir)
{return GetCurrentDirectory(lstrlen(pDir), pDir);}

BOOL SetCurDir(PTCHAR pDir)
{return SetCurrentDirectory(pDir);}

VOID GetSysInfo(PSHORT psProcArch, PINT piPageSize, PINT piProcMask,
								PINT piNumProcs, PINT piProcType, PINT piAllocGran,
								PSHORT psProcLevel, PSHORT psProcRev)
{
	SYSTEM_INFO	si;
	
	// No initialization necessary, just call.
	GetSystemInfo(&si);
	
	// Copy info to return vars.
	*psProcArch		= si.wProcessorArchitecture;
	*piPageSize		= si.dwPageSize;
	*piProcMask		= si.dwActiveProcessorMask;
	*piNumProcs		= si.dwNumberOfProcessors;
	*piProcType		= si.dwProcessorType;
	*piAllocGran	= si.dwAllocationGranularity;
	*psProcLevel	= si.wProcessorLevel;
	*psProcRev		= si.wProcessorRevision;
}

INT GetBootType()
{return GetSystemMetrics(SM_CLEANBOOT);}

BOOL GetScreenSize(PINT piSizeX, PINT piSizeY)
{
	BOOL	bRes = FALSE;

	// Get screen sizes.
	*piSizeX = GetSystemMetrics(SM_CXSCREEN);
	*piSizeY = GetSystemMetrics(SM_CYSCREEN);

	// Check to make sure we got real data.
	if(*piSizeX && *piSizeY) bRes = TRUE;

	return bRes;
}

BOOL GetWorkareaRect(PINT piLeft, PINT piTop, PINT piRight, PINT piBottom)
{
	RECT	rc;
	BOOL	bRes = FALSE;

	// No initialization necessary, just call.
	if((bRes = SystemParametersInfo(SPI_GETWORKAREA, 0, &rc, 0) != 0)) {
		*piLeft		= rc.left;
		*piTop		= rc.top;
		*piRight	= rc.right;
		*piBottom	= rc.bottom;
	}
	
	// Return our result code.
	return bRes;
}

BOOL IsSlowMachine()
{return GetSystemMetrics(SM_SLOWMACHINE);}

BOOL GetLoggedOnUserName(PTCHAR pName)
{
	DWORD	dwSize = lstrlen(pName);

	// For some reason, this call doesn't work like
	// other, similar calls. It returns the number of chars
	// copied to the dwSize parameter.
	return GetUserName(pName, &dwSize);
}

BOOL GetExtendedVersionInfo(PINT piMajor, PINT piMinor, PINT piBuild,
														PINT piPlatform, PTCHAR pSvcPack)
{
	BOOL					bRes = FALSE;
	OSVERSIONINFO	ovi;

	// Setup OSVERSIONINFO strct.
	ZeroMemory(&ovi, sizeof(ovi));
	ovi.dwOSVersionInfoSize = sizeof(ovi);

	// Make the call.
	if((bRes = GetVersionEx(&ovi)) != 0) {

		// If successful copy return values.
		*piMajor		= ovi.dwMajorVersion;
		*piMinor		= ovi.dwMinorVersion;
		*piBuild		= ovi.dwBuildNumber;
		*piPlatform	= ovi.dwPlatformId;
		strcpy(pSvcPack, ovi.szCSDVersion);
	}

	return bRes;
}

DWORD GetLastWinErr()
{
	// While we're at it, set our last error also.
	gdwErr = GetLastError();

	return gdwErr;
}

BOOL GetLastWinErrMsg(DWORD dwErr, PTCHAR pErrMsg)
{
	BOOL		bRes = FALSE;
	LPVOID	lpMsgBuf;

	// Check to see if we want to get the error ourselves.
	if(dwErr == 0) {
		gdwErr = GetLastError();
		dwErr = gdwErr;
	}

	// Get an error message from the system.
	if(FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM
		| FORMAT_MESSAGE_IGNORE_INSERTS, NULL, dwErr,
		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (LPTSTR)&lpMsgBuf, 0, NULL)) {

		// If successful, let caller know.
		bRes = TRUE;

		// Copy error message to return string and release system-allocated buffer.
		lstrcpy(pErrMsg, (LPCTSTR)lpMsgBuf);
		LocalFree(lpMsgBuf);
	}

	return bRes;
}

VOID SetLastWinErr(INT iErr)
{
	// While we're at it, set our error also.
	gdwErr = iErr;
	SetLastError(iErr);
}

BOOL SubclassButton(PTCHAR pText)
{
	HWND	hWnd = NULL;
	BOOL	bRes = FALSE;

	// First, see if we can find the button.
	if(hWnd = FindVeeChildWindow(pText)) {

		// Subclass it.
		if(gpfnPrevWndProc = (WNDPROC)SetWindowLong(hWnd, GWL_WNDPROC, (LONG)MyWndProc)) {

			// Note that we succeeded.
			ghWndButton = hWnd;
			bRes = TRUE;
		}
	}

	return bRes;
}

BOOL SetButtonText(PTCHAR pText)
{
	int		iLen;
	BOOL	bRes = FALSE;

	// Hmmm... notice that we can't set it to nothing: ""
	if(iLen = lstrlen(pText)) {
		
		// If we've previously allocated memory for text, kill it.
		if(gpButtonText) delete [] gpButtonText;
		
		// Allocate memory for new text.
		gpButtonText = new TCHAR[iLen + 1];
		
		// As long as we got memory, copy the desired text.
		if(gpButtonText) {
			lstrcpy(gpButtonText, pText);
			bRes = TRUE;
		}
	}

	return bRes;
}

BOOL SetButtonFont(INT iSize, PTCHAR pName)
{
	HDC			hDC;
	BOOL		bRes = FALSE;
	POINT		pt;
	LOGFONT lf;

	// If we've previously created a font for the button, get rid of it.
	if(ghButtonFont) DeleteObject(ghButtonFont);

	// This stuff comes pretty much directly from MFC's CFont::CreatePointFont.
	ZeroMemory(&lf, sizeof(lf));
	lf.lfCharSet = DEFAULT_CHARSET;
	lf.lfHeight = iSize;
	strcpy(lf.lfFaceName, pName);

	hDC = GetDC(NULL);
	pt.y = GetDeviceCaps(hDC, LOGPIXELSY) * lf.lfHeight;
	pt.y /= 720;
	DPtoLP(hDC, &pt, 1);
	POINT ptOrg = {0, 0};
	DPtoLP(hDC, &ptOrg, 1);
	lf.lfHeight = -abs(pt.y - ptOrg.y);
	ReleaseDC(NULL, hDC);
	if(ghButtonFont = CreateFontIndirect(&lf)) bRes = TRUE;

	return bRes;
}

VOID SetButtonBkgndColor(INT r, INT g, INT b)
{gcrBkgndColor = RGB(r, g, b);}

VOID SetButtonTextColor(INT r, INT g, INT b)
{gcrTextColor = RGB(r, g, b);}

VOID RedrawButton()
{
	HDC	hDC;

	// First, make sure we're subclassed.
	if(gpfnPrevWndProc && ghWndButton) {

		// Get a DC for the button & send a WM_PAINT message.
		hDC = GetWindowDC(ghWndButton);
		SendMessage(ghWndButton, WM_PAINT, (WPARAM)hDC, 0L);
		ReleaseDC(ghWndButton, hDC);
	}
}

VOID ClickButton()
{
	RECT	rc;
	POINT	pt;

	// First, make sure we're subclassed.
	if(gpfnPrevWndProc && ghWndButton) {

		// We'll need a phoney cursor location.
		GetClientRect(ghWndButton, &rc);
		pt.x = rc.right		- ((rc.right - rc.left) / 2);
		pt.y = rc.bottom	- ((rc.bottom - rc.top) / 2);

		// Post l-button down and up messages.
		PostMessage(ghWndButton, WM_LBUTTONDOWN, 0, MAKELONG(pt.x, pt.y));
		PostMessage(ghWndButton, WM_LBUTTONUP, MK_LBUTTON, MAKELONG(pt.x, pt.y));
	}
}

BOOL UnsubclassButton()
{
	BOOL	bRes = FALSE;
	
	// First, make sure we're subclassed.
	if(gpfnPrevWndProc && ghWndButton) {
		
		// Replace the new WNDPROC with the old one.
		gpfnPrevWndProc = (WNDPROC)SetWindowLong(ghWndButton, GWL_WNDPROC, (LONG)gpfnPrevWndProc);

		// Make darn sure it worked!
		if((LONG)gpfnPrevWndProc == (LONG)MyWndProc) {
			
			// If so, set everything back to normal.
			bRes						= TRUE;
			ghWndButton			= NULL;
			gpfnPrevWndProc	= NULL;

			// Delete any font we might have created.
			if(ghButtonFont) {
				DeleteObject(ghButtonFont);
				ghButtonFont = NULL;
			}

			// Release any memory we allocated for button text.
			if(gpButtonText) {
				delete [] gpButtonText;
				gpButtonText = NULL;
			}

		} else {

			// If not, we're in deep trouble.
			MessageBox(NULL, gszUnsubclsFailMsg, gszUnsubclsFailTitle, MB_OK | MB_ICONSTOP);
		}
	}

	return bRes;
}

BOOL VeeBeep(UINT uType)
{return MessageBeep(uType);}

BOOL ControlledBeep(DWORD dwFreq, DWORD dwDur)
{return Beep(dwFreq, dwDur);}

/////////////////////////////////////////////////////////////////////////////
// Helpers.
BOOL ChangeStyleBit(LONG lStyleBit, BOOL bNewState)
{
	LONG	lStyle;
	BOOL	bRes = FALSE;
	HRGN	hRegion;

	// Get the current window style.
	if(lStyle = GetWindowLong(ghWnd, GWL_STYLE)) {
		
		// OR in or AND out the requested bit.
		if(bNewState) lStyle |= lStyleBit;
		else lStyle &= (~lStyleBit);
		
		// Set new style.
		if(SetWindowLong(ghWnd, GWL_STYLE, lStyle)) {

			// If successful, redraw frame (the non-client area).
			bRes = TRUE;
			GetWindowRect(ghWnd, &grcWnd);
			hRegion = CreateRectRgn(grcWnd.left, grcWnd.top, grcWnd.right, grcWnd.bottom);
			SendMessage(ghWnd, WM_NCPAINT, (WPARAM)hRegion, 0L);
			DeleteObject(hRegion);
		}
	}

	return bRes;
}

HWND FindVeeChildWindow(PTCHAR pText)
{
	HWND	hFoundWnd = NULL;

	// If there's a window chain, get rid of it.
	if(gpWinRoot) DeleteWinNodes(gpWinRoot);

	// Start a window chain.
	if(gpWinRoot = AllocWinNode()) {

		// Enumerate windows.
		EnumChildWindows(ghWnd, (WNDENUMPROC)EnumChildWindowProc, (LPARAM)gpWinRoot);

		// Look for our text.
		PWINNODE p = gpWinRoot;
		do {
			
			// If there's text allocated.
			if(p->pText) {
				
				// See if it's the text we're looking for.
				if(lstrcmp(p->pText, pText) == 0) {
					
					// If so, note the handle & exit the loop.
					hFoundWnd = p->hWnd;
					break;
				}
			}
			
			// Check the next node.
			p = p->pNext;
		
		} while(p);
	}

	// Get rid of the chain we just allocated (AND set root pointer to NULL).
	DeleteWinNodes(gpWinRoot);
	gpWinRoot = NULL;

	// This is the handle we found.
	return hFoundWnd;
}

/////////////////////////////////////////////////////////////////////////////
// How about a subclassing example?
LRESULT CALLBACK MyWndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	HDC			hDC;
	RECT		rc;
	HBRUSH	hBr;

	// This is where we'll handle messages for our window.
	switch(uMsg) {
	
		// When we're supposed to erase the button background, get the rectangle
		// to fill. Create a brush of the appropriate color and fill the rect.
		// Delete the brush and tell Windows we processed this message.
		case WM_ERASEBKGND:
			GetClientRect(hWnd, &rc);
			hBr = CreateSolidBrush(gcrBkgndColor);
			FillRect((HDC)wParam, &rc, hBr);
			DeleteObject(hBr);
			return 1L;
		break;

		// When we're supposed to paint our button, we're going to cheat a little.
		case WM_PAINT:
			HGDIOBJ			hOldFont;
			PAINTSTRUCT ps;

			// First, get our paint struct. Normally, we'd use this.
			hDC = BeginPaint(hWnd, &ps);

			// Things don't work quite the way they're supposed to though, so
			// we're going to use the basic client rectangle as coordinates.
			GetClientRect(hWnd, &rc);

			// Set colors.
			SetBkColor(hDC, gcrBkgndColor);
			SetTextColor(hDC, gcrTextColor);

			// If we've changed our font, select it.
			// Draw the text & select the previous font (if we changed it).
			if(ghButtonFont) hOldFont = SelectObject(hDC, ghButtonFont);
			DrawText(hDC, gpButtonText, -1, &rc, DT_SINGLELINE | DT_VCENTER | DT_CENTER);
			if(ghButtonFont) SelectObject(hDC, hOldFont);

			// Validate the rectangle just painted.
			ValidateRect(hWnd, NULL);

			// Call EndPaint and tell Windows we processed this message.
			EndPaint(hWnd, &ps);
			return 0L;
		break;

		// All other messages get passed to the previous window procedure
		// or DefWindowProc, as decided by Windows.
		default:
			return CallWindowProc((WNDPROC)gpfnPrevWndProc, hWnd, uMsg, wParam, lParam);
		break;
	}
}

/////////////////////////////////////////////////////////////////////////////
// Child window enumeration proc.
BOOL CALLBACK EnumChildWindowProc(HWND hWnd, LPARAM lParam)
{
	PTCHAR		pText;
	PWINNODE	p = (PWINNODE)lParam;

	// Get four of this window's extra data bytes.
	// Interpreting them as a pointer to char, see if this pointer is non zero.
	if(pText = (PTCHAR)(GetWindowLong(hWnd, 0))) {

		// Call it a pointer to a window title
		// minus 0xd0 and see if the first character is alphanumeric.
		pText += 0xd0;
		if(isalpha(*pText)) {
		
			// Let's assume we've found a window title.
			// Walk to the bottom of the WINNODE chain.
			while(p->pNext) p = p->pNext;

			// Fill this node.
			p->hWnd = hWnd;
			if(p->pText = new TCHAR[lstrlen(pText) + 1]) strcpy(p->pText, pText);

			// If we can't allocate memory for a new node, stop the enumeration.
			if(!(p->pNext = AllocWinNode())) return FALSE;
		}
	}

	// Tell Windows to continue enumerating windows 'til we have them all.
	return TRUE;
}

PWINNODE AllocWinNode()
{
	// Get a new node.
	PWINNODE p = new WINNODE;
	
	// If we got it, initialize it.
	if(p) ZeroMemory(p, sizeof(WINNODE));
	
	// Return it's address.
	return p;
}

VOID DeleteWinNodes(PWINNODE pNode)
{
	// Recursively delete nodes.
	if(pNode->pNext) DeleteWinNodes(pNode->pNext);

	// Delete any text allocation first.
	if(pNode->pText) delete [] pNode->pText;

	// Then delete the node.
	delete pNode;
}
