// ****************************************************************************
//
// File Name:    preform.c
//
// Description: Preform a selected function.  Request information necessary 
// for the given command and then execute it.
//
//
//  ORIGINATOR: Escort Memory Systems
//
//  HISTORY
//    who     when     what
// -------- --------- ----------------------------------------------------
//
// ****************************************************************************

#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>
#include <time.h>      // For down count display.
#include <ctype.h>     // Lower case conversion.
#include "typedef.h"   // General typedef and defines.
#include "handheld.h"
#include <mem.h>
#include "preform.h"
#include "hms800.h"
#include "comm.h"


#define HMS_HEX_LINES   20 // Bytes displayed on a line in Hexadecimal format.
#define HMS_DEC_LINES   15 // Bytes displayed on a line in decimal format.
#define HMS_ASCII_LNS   70 // Bytes displayed on a line in ASCII format.
#define HMS_BIN_LINES    8 // Bytes displayed on a line in Hexadecimal format.
#define DISPLAY_LINES   24 // Number of data lines to display on one screen full.
#define DISPLAY_WIDTH   80 // Maximum Number of characters on one line if display.

#define HS200_HEX_LNS   25 // Bytes displayed on a line in Hexadecimal format.
#define HS200_DEC_LNS   15 // Bytes displayed on a line in decimal format.
#define HS200_ASCII     70 // Bytes displayed on a line in ASCII format.
#define HS200_BIN_LNS    8 // Bytes displayed on a line in Hexadecimal format.

#define HS208_HEX_LNS   20 // Bytes displayed on a line in Hexadecimal format.
#define HS208_DEC_LNS   15 // Bytes displayed on a line in decimal format.
#define HS208_ASCII     70 // Bytes displayed on a line in ASCII format.
#define HS208_BIN_LNS    8 // Bytes displayed on a line in Hexadecimal format.

#define HS232_HEX_LNS   20 // Bytes displayed on a line in Hexadecimal format.
#define HS232_DEC_LNS   15 // Bytes displayed on a line in decimal format.
#define HS232_ASCII     70 // Bytes displayed on a line in ASCII format.
#define HS232_BIN_LNS    8 // Bytes displayed on a line in Hexadecimal format.

#define DOWN_DISPLAY   FALSE 
#define TAG_START        15 // Initial down count value when waiting for a tag.
#define TAG_FILLING       3 // After tag found, every tag fill command for hole tag.
#define TAG_CONTINUAL     1 // down count timeout value when in continual read.
#define MAX_HEX_DIG       2 // Maximum number of hexadecimal digits in a BYTE.
#define MAX_BIN_DIG       8 // Maximum number of binary digits in a BYTE.
#define MAX_DEC_DIGITS    5 // More then maximum number of decimal digits input.

#define TOO_MANY_ERRORS   10   // Too many read errors on this tag.

#define NUM_REGIONS        1   // One region block.
#define LAST_PROT_OFF   0x0A   // Offset value of last protected byte to hide in file.

#define HS_ID_START        1   // Tag address of first BYTE of HS tag ID.
#define HS_ID_END          2   // Seconds and last Tag ID byte address.

#define OUT_LIMIT           128  // Outut buffer array size.

enum data_type display_type;   // Display format chosen.
enum ant_type antenna_type = NOT_VALID; // "flag" value for antenna type selected.
WORD mem_size = HMS_MEM_SIZE;       // Size of selected antenna type tag.
BYTE max_data_pkt = HMS827_MAX_DATA;  // Maximum data bytes size for selected antanna.

static __far BYTE tag_io_buf[HMS_MEM_SIZE]; // Working IO buffer for all tag BYTEs.

static time_t last_time;            // Value returned by time() last time.
static int display_val;             // Down count  display value.
static BYTE finished;               // Local BOOLEAN Not finished with mifare command.

// Copy, hold and write an entire tag working variables.
static BYTE tag_buf[HMS_MEM_SIZE];  // Hold all BYTEs from a tag.
WORD start_adr = TOO_BIG_VAL;// Begin coping from this address.
WORD stop_adr = TOO_BIG_VAL; // Last address byte to be coping.

// Below working variables for non-contiguous I/O routines.
static BYTE *address_ptr;   // Index the output buffer to build command packet.
static WORD address_value;  // Hold the input address.
static WORD nc_read_size;   // Number of bytes in for Non-contiguous Read.

// ****************************************************************************
// Preform a Block Read command. 
// Input:        Begin reading at this start address in the tag.
//               Number of bytes to be read from tag.
//               Where to place the received data.
// Return:       Read STATUS.
// Side effects: None.
static BYTE Read_command(WORD start_add, WORD data_size, BYTE *rx_data)
{
   BYTE  status;     // Mifare tag interface status value.

   BLReadTx(start_add, data_size, MIFARE_TIME_OUT); 
   status = BLReadRx(rx_data, data_size);
   return(status);
}

// ****************************************************************************
// Preform a Block Write command. 
// Input:        Begin write at this start address in the tag.
//               Number of bytes to be read from tag.
//               Pointer to the bytes to be written.
//               BOOLEAN - TRUE, using proteced mode block writes.
//                       - FALSE, normal block write.
// Return:       Read STATUS.
// Side effects: None.
static BYTE Write_command(WORD start_add, WORD data_size, BYTE *tx_data, BYTE prt_mode)
{
   BYTE  status;     // Mifare tag interface status value.

   if (prt_mode)
   {
      pro_BLWriteTx(start_add, data_size, MIFARE_TIME_OUT, tx_data);
      status = pro_BLWriteRx();
   } else
   {
      BLWriteTx(start_add, data_size, MIF_WRITE_TI, tx_data);
      status = BLWriteRx();
   }
   return(status);
}

// ****************************************************************************
// What is the value of the decimal number in the digit string pointed to?
// That value is placed in the WORD pointed to by value parameter.
// The decimal number can be terminated with spaces (' ') or a carriage return.
// If the digits overflow a WORD, then what *value points to is set to zero (0)
// and this function returns FALSE.
//
// Input:        Pointer to where digit string is located.
//               Pointer to a WORD where the value from the string is place,
//               unless there is an overflow and FALSE is returned.
// Return:       BOOLEAN - TRUE,  Valid decimal number input, with no overflow.
//                         FALSE, digits in the string will overflow a WORD. 
// Side effects: The WORD value points to will be altered.
static BYTE dec_word_val(BYTE* digit_string, WORD *value) 
{
   unsigned long int incom_value;

   *value = incom_value = 0;
   while ((*digit_string >= '0') AND (*digit_string <= '9'))
   {
      incom_value = (incom_value * 10) + *digit_string - '0'; // Add in new value.
      if (incom_value > TOO_BIG_VAL)  // This string overflowed.
         return(FALSE);
      digit_string++;        // Count another valid digit entered.
   }
   *value = (WORD)incom_value;
   return(TRUE);
}

// ****************************************************************************
// Is the string pointed to by the passed pointer all eight digits for a 
// binary BYTE?  Assuming the string was pre-fille with '.' or ' ', those
// eight (8) characters should be replaced with 1 or 0 before this function
// is called.  This is to check if all eight (8) binary digits have been entered.
//
// Input:        Pointer to a string of binary digits.
// Return:       BOOLEAN - TRUE,  all eight (8) binary digits are in the string.
//                         FALSE, not all eight (8) binary digits.
// Side effects: None.
static WORD eight_binary_digits(BYTE *binary_string)
{
   WORD ii;       // Local loop counter.
   
   for (ii = MAX_BIN_DIG; ii; ii--)
   {
      if ((*binary_string < '0') OR (*binary_string > '1'))
         return(FALSE);     // Non-binary digit found.  Thus not all binary.
      binary_string++;      // Look at the next digt.
   }
   return(TRUE);            // Must be all binary.
}

// ****************************************************************************
// For use with the Del key when entering number on the screen.  The first
// digit in the digit string pointed to by the passed parameter is deleted.
// The remaining digits are shifted to the left none character space.
// These remaining digit are then printed on the screen and then the cursor is
// backed up over them, to leave the cursor in the same position.
//
// Input:        Pointer to the string of digits.
// Return:       None
// Side effects: String pointed to by the digit string pointer looses a digit.
static void del_digit(BYTE *dig_str)
{
   WORD num_digits;     // Number of digits after the deleted one to be shifted.
   WORD prt_count;      // Loop counter for printing the shifted digits.
   BYTE *dig_ptr;       // Used to index the digit string.
   
   num_digits = 0;      // May not shift any digits.
   dig_ptr = dig_str;
   while ((*dig_ptr >= '0') AND (*dig_ptr <= '9')) // While in the digits
   {                                               // Shift the next one down.
      *dig_ptr = *(dig_ptr + 1);
      dig_ptr++;
      num_digits = num_digits + 1;  // Another digit shifted.
   }
   dig_ptr = dig_str;
   for (prt_count = num_digits; prt_count; prt_count--)
      if (*dig_ptr NE CR)           // Last digit handled?
         putch(*dig_ptr++);         // No, echo the digit.
      else
         putch(' ');                // Yes, do not echo a carriage return.
   for ( ; num_digits; num_digits--)
      putch(BACK_SPACE);            // Get the cursor back to where it began.
}


// ****************************************************************************
// To read in a decimal number, the digits must be entered. That is what this 
// routine does. The digit input string will be terminated with a carriage return.
// This routine will only accept and echo characters of 0 to 9.
// Assumed the ASCII string of more than MAX_DEC_DIGITS BYTEs (one for the CR).
//
// Input:        Pointer to where the digit input string should be placed.
//                  It is assumed to have at least MAX_DEC_DIGITS+1 BYTES of space.
//               The maximum value allowed for this input.
//               The minimum value allowed for this input.
// Return:       BOOLEAN - TRUE,  characters input.
//                         FALSE, ESE entered to exit.
// Side effects: Length BYTES at the ASCII string might be altered.
static BYTE inp_decimal(BYTE* digits_here, WORD min_value, WORD max_value) 
{
   WORD  cur_value;           // Hold the current value being read in.
   WORD  dig_pos;             // The string position (index) of the next character.
   WORD  num_digts;           // Number of digits in the string, Maximum entered.
   BYTE  ascii_string[MAX_DEC_DIGITS + 2];  // Hold digits being input.
   BYTE  inp_char;            // Last character input.
   
   num_digts = dig_pos = cur_value = 0;   // No valid digits input yet.
   memset(digits_here, ' ', MAX_DEC_DIGITS);  // Fill with spaces.
   digits_here[MAX_DEC_DIGITS] = CR;    // Be sure it is carriage return terminated.
   do
   {
      if ((inp_char = getch()) EQ ESC)  // This echos the character.
         return(FALSE);           // Exit on ESC.  Confused user?
      if (inp_char EQ ARROW_HI_BYTE) // It might be a valid control key.
      {  // Then, if the next input makes it a valid control continue the do loop.
         if ((inp_char = getch()) EQ LEFT_ARROW)
         {
            if (dig_pos) // Some digits have been input to back up over.
            {
               putch(BACK_SPACE);         // Back up on the screen.
               dig_pos = dig_pos - 1;     // Backing up over a character.
            } else
               putch(BELL);                  // ERROR beep, if can.
         } else if (inp_char EQ RIGHT_ARROW)
         {
            if ((dig_pos < MAX_DEC_DIGITS - 1) AND (dig_pos < num_digts))
               putch(digits_here[dig_pos++]); // Print again what is in the buffer.
            else
               putch(BELL);                  // ERROR beep, if can.
         } else if (inp_char EQ DELETE)
         {
            if (num_digts AND (num_digts NE dig_pos)) 
               num_digts = num_digts - 1;
            del_digit(&digits_here[dig_pos]);  // and update the display.
         } else
            putch(BELL);                  // ERROR beep, if can.
      } else if (inp_char EQ BACK_SPACE)
      {
         if (dig_pos) // Some digits have been entered to back up over.
         {
            putch(BACK_SPACE);            // Backup.
            dig_pos = dig_pos - 1;        // New cursor position.
            if (num_digts AND (num_digts NE dig_pos)) 
               num_digts = num_digts - 1;
            del_digit(&digits_here[dig_pos]);  // and update the display.
         } else
            putch(BELL);                  // ERROR beep, if can.
      } else if (inp_char NE CR)          // A carriage return is used to terminate
      {                                   // The CR terminator will be added later.
         if (dig_pos < MAX_DEC_DIGITS)
         {
            if (inp_char >= '0' AND inp_char <= '9') 
            {  // Valid character input. Put it in string copy and check for
               memcpy(ascii_string, digits_here, MAX_DEC_DIGITS + 1);
               ascii_string[dig_pos] = inp_char; // Put in buffer for to check it.
               if (dec_word_val(ascii_string, &cur_value))
               {
                  if (cur_value <= max_value)
                  {
                     putch(inp_char);           // Echo what was just input.
                     digits_here[dig_pos++] = inp_char; // Save in input buffer.
                     if (dig_pos > num_digts)
                        num_digts = dig_pos;    // Sting getting longer.
                  } else
                     putch(BELL);               // ERROR beep, if can.
               } else // The new digit entered would make an overflow.
                  putch(BELL);                  // ERROR beep, if can.
            } else
               putch(BELL);                     // ERROR beep, if can.
         }
      } else if ((cur_value < min_value) AND (num_digts NE 0)) // Invalid ENTER key?
          putch(BELL);                           // ERROR beep, if can.
      dec_word_val(digits_here, &cur_value);     // Set to current value in string.
      if ((inp_char EQ CR) AND (num_digts EQ 0)) // No digits just ENTER.
         cur_value = min_value;                  // force exit.
   } while ((inp_char NE CR) OR (cur_value < min_value)); // Until CR and big enough.
   digits_here[num_digts] = CR;  // Terminate with a carriage return.
   return(TRUE);
}

// ****************************************************************************
// To read in a number the digits must be entered. That is what this routine does.
// It actually just reads in a string of bytes input at the keypad.  This
// is to support exiting with ESC and use of the back space or left and right
// arrow keys.  The digit input string will be terminated with the carriage return.
// It will only accept and echo characters for the base being input for.
// Assumed the ASCII string has points to at least (number_length + 1) data bytes.
// Hexadecimal, decimal, and binary are all supported.  Although, (because of
// the Del key) this should only be used for hexadecimal and binary input.
//
// Input:        Pointer to where the digit input string should be placed.
//               Maximum number of digits to be input for this number.
//                  The *ASCII string must have one more byte, for the carriage return.
//               Base (HEX, DEC,or BIN ) the digits are being input for.
//               The character to 'fill' the empty digit string with, before digits.
// Return:       BOOLEAN - TRUE,  characters input.
//                         FALSE, ESE entered to exit.
// Side effects: Length BYTES at the ASCII string might be altered.
static BYTE inp_digstring(BYTE* ascii_string, WORD number_length, 
   enum data_type base, BYTE fill_char) 
{
   int   num_digs;     // Number of digits that have been entered.
   WORD  dig_pos;      // The string position (index) of the next character.
   BYTE  inp_char;     // Last character input.
   
   num_digs = dig_pos = 0;   // No valid digits input yet.
   memset(ascii_string, fill_char, number_length); // Fill with spaces.
   ascii_string[number_length] = CR; // Terminate with a carriage return. 
   do
   {
      if ((inp_char = getch()) EQ ESC)  // This echos the character.
         return(FALSE);           // Exit on ESC.  Confused user?
      if (inp_char EQ ARROW_HI_BYTE)
      {
         if ((inp_char = getch()) EQ LEFT_ARROW)
         {
            if (dig_pos) // Some digits have been input to back up over.
            {
               putch(BACK_SPACE);            // Back up on the screen.
               dig_pos = dig_pos - 1;        // Backing up over a character.
            } else
               putch(BELL);                  // ERROR beep, if can.
         } else if (inp_char EQ RIGHT_ARROW)
         {
            if (dig_pos < number_length - 1)
               putch(ascii_string[dig_pos++]); // Print again what is in the buffer.
            else
               putch(BELL);                  // ERROR beep, if can.
         } else if (inp_char EQ DELETE)
         {
            if (dig_pos < number_length)  // At the end of the bigest number?
            {                             // NO, there is something to delete.
               if (ascii_string[dig_pos] NE fill_char) // Is a digit being deleted?
                  num_digs = num_digs - 1; // A digit deleted.
               ascii_string[dig_pos] = fill_char; // Last character is gone.
               putch(fill_char);           // Erase from screen last character.
               putch(BACK_SPACE);          // Backup over the ' ' just printed.
            }
         } else
            putch(BELL);                  // ERROR beep, if can.
      } else if (inp_char EQ BACK_SPACE)
      {
         if (dig_pos) // Some digits have been input to back up over.
         {
            putch(inp_char);                   // Echo what was just input.
            dig_pos = dig_pos - 1;             // Backing up over a character.
            if (ascii_string[dig_pos] NE fill_char) // Is a digit being deleted?
               num_digs = num_digs - 1;        // A digit deleted.
            ascii_string[dig_pos] = fill_char; // Last character is gone.
            putch(fill_char);                  // Erase from screen last character.
            putch(BACK_SPACE);                 // Backup over the ' ' just printed.
         } else
            putch(BELL);                  // ERROR beep, if can.
      } else if (inp_char NE CR)
      { // Leave at least one space (' ') to terminate the number.
         if (dig_pos < number_length)
         {
            switch (base)
            {
            case HEX:
               inp_char = toupper(inp_char); // Convert to upper case.
               if ((inp_char >= '0' AND inp_char <= '9') OR
                  (inp_char >= 'A' AND inp_char <= 'F'))
               {
                  putch(inp_char);                  // Echo what was just input.
                  ascii_string[dig_pos++] = inp_char; // Save in input buffer.
                  num_digs = num_digs + 1;   // Count another digit entered.
               } else
                  putch(BELL);               // ERROR beep, if can.
               break;

            case DEC:
               if (inp_char >= '0' AND inp_char <= '9') 
               {
                  putch(inp_char);                  // Echo what was just input.
                  ascii_string[dig_pos++] = inp_char; // Save in input buffer.
                  num_digs = num_digs + 1;   // Count another digit entered.
               } else
                  putch(BELL);               // ERROR beep, if can.
               break;

            case BIN:
               if (inp_char >= '0' AND inp_char <= '1') 
               {
                  putch(inp_char);                  // Echo what was just input.
                  ascii_string[dig_pos++] = inp_char; // Save in input buffer.
                  num_digs = num_digs + 1;   // Count another digit entered.
               } else
                  putch(BELL);               // ERROR beep, if can.
               break;
               
            default:
               return(FALSE);    // ERROR! this should never happen
            }
         }
      } else if (base EQ BIN) // It is a CR, but Binary must input all digits.
         if ((NOT eight_binary_digits(ascii_string)) AND  // If all binary digitse
            (num_digs NE 0))    // And some digits have been entered.
         {   
            inp_char = 0;       // Not all a 1 or 0, then not done. Ignore CR.
            putch(BELL);        // ERROR beep, if can.
         }
   } while (inp_char NE CR); 
   if (num_digs <= 0)
      ascii_string[0] = CR;   // ENTER key with no digits.
   printf("\n");      
   return(TRUE);
}

// ****************************************************************************
// Input a decimal number, character by character. 
// sscanf() would be used, but that waits for a CR.  
// ESC must be detected, to exit.
// ENTER alone should also be recognized as a special case.
// As a result, inp_num() is returning an unsigned number that will have a value
// that will fit in a signed number.  ALL valid numbers (not a flag)
// will have the sign bit cleared. Flag values, for things like ESC or ENTER,
// will have the sign bit set (0x8000).
//
// Input:        The maximum value allowed for this input.
// Return:       Number input, TOO_BIG_VAL if ESC was entered to exit,
//               or ENTER_ALONE if no digits entered, but the ENTER key was pressed.
// Side effects: NONE.
WORD inp_num(WORD max_value) 
{
   BYTE  ascii_number[MAX_DEC_DIGITS + 2];  // Hold digits being input.
   WORD  this_value;          // Build the value being input here.
   
   this_value = 0;            // Initialize working variables.
   if (NOT inp_decimal(ascii_number, 0, max_value))
       return(TOO_BIG_VAL);   // Use is trying to ESCape out.
   printf("\n");              // Echo the terminating carriage return.
   if (ascii_number[0] EQ CR) // No digits, only the ENTER key.
      return(ENTER_ALONE);    // Thus, done with entering data.
   if (dec_word_val(ascii_number, &this_value))
      return(this_value);     // Valid decimal number entered. Return it!
   return(0);                 // Should never get here, but be safe.
}

// ****************************************************************************
// Input a hexadecimal number, character by character. sscanf() would be used, 
// but that waits for a CR.  ESC must be detected, to exit.
//
// Input:        NONE.
// Return:       Number input, TOO_BIG_VAL if ESC was entered to exit,.
//               or ENTER_ALONE if no digits entered, but the ENTER key was pressed.
// Side effects: NONE.
WORD inp_hex(void) 
{
   WORD  incom_value;    // Build the value being input here.
   BYTE  ascii_number[MAX_HEX_DIG + 2];  // Hold digits being input and a CR.
   BYTE* digit_ptr;      // Point to where the next digit goes.

   incom_value = 0;
   if (NOT inp_digstring(ascii_number, MAX_HEX_DIG, HEX, ' '))
      return(TOO_BIG_VAL);     // Use is trying to ESCAPE out.
   digit_ptr = ascii_number;   // Point at the character just input.
   if (*digit_ptr EQ CR)       // No digits, only the ENTER key.
      return(ENTER_ALONE);     // Return to caller for a try again, or default.
   while (*digit_ptr NE CR)    // Loop until done and time to returns.
   {                          
      if ((*digit_ptr >= '0') AND (*digit_ptr <= '9'))  // Another digit entered?
         incom_value = (incom_value << 4) + *digit_ptr - '0'; // Add in new value.
      else if ((*digit_ptr >= 'A') AND (*digit_ptr <= 'F'))   // Hex dig?
         incom_value = (incom_value << 4) + *digit_ptr - 'A' + 0x0A; // Update value.
      digit_ptr++;        // Count another valid digit entered.
   }
   return(incom_value);   // Done imputing, return the input value.
}

// ****************************************************************************
// Input a binary number, character by character. sscanf() would be used, 
// but that waits for a CR.  ESC must be detected, to exit.
//
// Input:        NONE.
// Return:       Number input, TOO_BIG_VAL if ESC was entered to exit,
//               or ENTER_ALONE if no digits entered, but the ENTER key was pressed.
// Side effects: NONE.
WORD inp_bin(void) 
{
   WORD incom_value;     // Build the value being input here.
   BYTE ascii_number[MAX_BIN_DIG + 2];  // Hold digits being input + CR, terminator.
   BYTE *digit_ptr;      // Point to where the next digit goes.
   WORD ii;              // Local loop counter, for all eight digits.

   incom_value = 0;
   printf("........\10\10\10\10\10\10\10\10"); // Dots to become binary digits.
   if (NOT inp_digstring(ascii_number, MAX_BIN_DIG, BIN, '.'))
       return(TOO_BIG_VAL); // Use is trying to ESCAPE out.
   digit_ptr = ascii_number;        // Point at the character just input.
   if (*digit_ptr EQ CR)       // No digits, only the ENTER key.
      return(ENTER_ALONE);     // Return to caller for a try again, or default.
   for (ii = MAX_BIN_DIG; ii; ii--)
      incom_value = (incom_value << 1) + *digit_ptr++ - '0'; // Add in new value.
   return(incom_value);   // Done imputing, return the input value.
}

// ****************************************************************************
// Input a decimal number, character by character, for a tag range.
// sscanf() would be used, but that waits for a CR.  ESC must be detected, to exit.
// A NULL entry (Enter alone with no digits) must be detected to indicate entire tag.
//
// Input:        The tags valid staring Address. HS tags can not write to address 0.
//               Last valid tag address for this range.  Protected and/or tag type.
// Return:       Number input, 
//               or TOO_BIG_VAL to exit,
//               or ENTER_ALONE if no digits entered, but the ENTER key was pressed.
// Side effects: NONE.
static WORD inp_tag_range(WORD tag_start, WORD tag_end) 
{
   BYTE  ascii_number[MAX_DEC_DIGITS + 2];  // Hold digits being input.
   WORD  this_value;          // Build the value being input here.
   
   this_value = 0;            // Initialize working variables.
   if (NOT inp_decimal(ascii_number, tag_start, tag_end))
       return(TOO_BIG_VAL);   // Use is trying to ESCape out.
   printf("\n");              // Echo the terminating carriage return.
   if (ascii_number[0] EQ CR) // No digits, only the ENTER key.
      return(ENTER_ALONE);    // Thus, done with entering data.
   if (dec_word_val(ascii_number, &this_value))
      return(this_value);     // Valid decimal number entered. Return it!
   return(0);                 // Should never get here, but be safe.
}

// ****************************************************************************
// Delay while something is being displayed.
//
// Input:        None.
// Return:       None.
// Side effects: None.
static void do_delay(time_t delay_time) 
{
   time_t start_time, now_time; // Hold second count for display.

   time(&start_time);       // Get current second count form the runtime.
   time(&now_time);
   while ((start_time + delay_time) >= now_time)
      time(&now_time);      // Get new second value.
}

// ****************************************************************************
// Delay while something is being displayed.
//
// Input:        None.
// Return:       None.
// Side effects: None.
static void display_delay(void) 
{
   do_delay(DISPLAY_TIME);  // Do the display delay 
}

// ****************************************************************************
// Delay while something is being displayed.
//
// Input:        None.
// Return:       None.
// Side effects: None.
void display_delay_sht(void) 
{
   do_delay(SHORT_DELAY);  // Do the display delay 
}

// ****************************************************************************
// Request and save the beginning (lower) and ending (upper) bound of tag data
// to be read. The upper bound is gotten from taking the staring Address and
// adding the length that the user specified.  A length that would exceed
// the end of tag is not taken, and it will be requested again.
//
// Input:        Pointer to start and stop tag addresses to be updated.
//               The tags valid staring Address. HS tags can not write to address 0.
//               The tags valid higher bound. For protected writing.
// Return:       BOOLEAN - TRUE, got a valid range.
//                       - FALSE, Not a valid range, the user gave up.
// Side effects: None.
static BYTE req_tag_range(WORD* lower, WORD* upper, WORD tag_start, WORD high_bound)
{
   WORD length;  // Length (number of bytes) for the tag range.
     
   if (tag_start EQ mem_size)      // Entire tag protected?
   {
      printf("Nothing between\n%u and %u.\n", tag_start, mem_size - 1);
      display_delay();
      return (FALSE);
   }
   printf("Starting address (%u to %u),\nor ENTER for entire %u to %u: ", 
      tag_start, high_bound, tag_start, high_bound);
   if ((*lower = inp_tag_range(tag_start, high_bound)) EQ TOO_BIG_VAL)
      return(FALSE);          // So return to exit this function.
   if (*lower EQ ENTER_ALONE) // Requesting entire tag?
   {
      *lower = tag_start;     // From the first valid tag address.
      *upper = high_bound;    // To the end of the tag.
   } else
   {
      printf("\nNumber of bytes?\nENTER for rest of tag, (1 to %u): ",
         high_bound + 1  - *lower);
      if ((length = inp_tag_range(1, high_bound + 1  - *lower)) EQ TOO_BIG_VAL)
         return(FALSE);          // So return to exit this function.
      else if (length EQ ENTER_ALONE) // Requesting to the end of tag?
         *upper = high_bound;  // To the end of the tag.
      else 
         *upper = *lower + length - 1;
   }
   return(TRUE);      // Got a valid and desired address range.
}

// ****************************************************************************
// Get the line length for the current display format. 
//
// Input:        Implied, display_type.
// Return:       None.
// Side effects: None.
static BYTE format_size(void) 
{
   switch (display_type)
   {
   case HEX:
      return(HMS_HEX_LINES);

   case DEC:
      return(HMS_DEC_LINES);

   case ASCII:
      return(HMS_ASCII_LNS);

   case BIN:
      return(HMS_BIN_LINES);
   }
   return(0);  // Never get here, this is just to make the compiler happy..
}


// ****************************************************************************
// Print, on the console, the current data byte.  Display in it the format
// chosen from the display options menu.
//
// Input:        BYTE to be printed.
// Return:       NONE.
// Side effects: NONE.
static void print_byte(BYTE disp_data) 
{
WORD ii;      // Local loop counter.

   switch (display_type)
   {
   case HEX:
      printf(" %02X", disp_data);
      break;

   case DEC:
      printf(" %3d", disp_data);
      break;

   case ASCII:
      printf("%c", disp_data);
      break;

   case BIN:
      printf(" ");  // Space between numbers.
      for (ii = 0x80; ii; ii = ii >> 1)
         if (ii & disp_data)
            putch('1');
         else
            putch('0');
      break;
   }
}


// ****************************************************************************
// Print the data for a line being dsplayed, in the current format.
// The tag address leader is also printed from this routines.
//
// Input:        Pointer to data to be printed.
//               Number of bytes in the array pointed to.
//               Address in the tag of the data being printed. 
// Return:       None.
// Side effects: None.
static void print_data_line(BYTE* data_ptr, WORD num_bytes, WORD data_adr) 
{
   WORD ii;              // Local loop counter.
   
   putch(CR);  // Be positive to be at the beginning of line.
   printf("A%3d:", data_adr);
   if (display_type EQ ASCII)      // If displaying ASCII, then put a 
      putch(' ');                  // space after the :. Because no spaces later.
   for(ii = num_bytes; ii; ii--)   // for data_adr bytes.
      print_byte(*data_ptr++);
   printf("\n");                  // final CR behind last data.
}

// ****************************************************************************
// Initialize the Display a down counter number. This is for down_count()
//
// Input:        Implied - set_doun() has been called to Initialization.
// Return:       BOOLEAN - TRUE, still down counting Borlands second counter
//                         FALSE, Done waiting.
// Side effects: NONE, outside down count.
//               finished is Initialized to FALSE.
static void start_down(WORD start_value) 
{
   
   display_val = start_value;  // Begin display time.
   time(&last_time);
   finished = FALSE;           // The mikron interface not finished.
}

// ****************************************************************************
// Display a down counter number:
// Do not print new line but back up so the next thing printed prints over
// the number just displayed. It is intended to have the cursor at a location
// and then repetitively call this routine while waiting.
// It looks for a 'change' in the runtime time_t routine and decrements display
// counter.
//
// Input:        Implied - set_doun() has been called to initialization.
// Return:       BOOLEAN - TRUE, still down counting Borlands second couter
//                         FALSE, Done waiting.
// Side effects: NONE, outside .
static BYTE down_count(void) 
{
   time_t now_time;
   time(&now_time);       // Get current second count form the runtime.

   if ((display_val = display_val - (now_time - last_time)) < 0)
      display_val = 0;    // NO underflow.
   last_time = now_time;  // Remember when here last
#if DOWN_DISPLAY
   printf("%4d", display_val);
   putch(BACK_SPACE);
   putch(BACK_SPACE);
   putch(BACK_SPACE);
   putch(BACK_SPACE);
#endif
   return(display_val);
}

// ****************************************************************************
// Ask the user if the current command should be executing again.
//
// Input:        NONE.
// Return:       BOOLEAN - TRUE, YES, the ESCAPE key has just been pressed.
//                         FALSE, NO, the eSCAPE key has NOT been pressed.
// Side effects: NONE.
static BYTE esc_entered(void) 
{
   if (kbhit())
      if (getch() EQ ESC)
         return(TRUE);     // Escape key enterd for the user to giving up.
   return(FALSE);          // Escape key not pressed.
}               

// ****************************************************************************
// Ask the user if the current command should be executing again.
//
// Input:        NONE.
// Return:       BOOLEAN - TRUE, YES, the user desires to do it again.
//                         FALSE, NO, the user is finished.
// Side effects: NONE.
static BYTE ask_again(void) 
{
   printf("\nENTER to repeat.\n");
   return(getch() EQ CR);
}

// ****************************************************************************
// Print the text string for the Error codes in the Reader/Writer manual.
//
// Input:        NONE.
// Return:       BOOLEAN - TRUE,  characters input for a number.
//                         FALSE, ESE entered to exit.
// Side effects: number length BYTES at the ASCII string might be altered.
void print_code(BYTE error_num) 
{
   switch (error_num)
   {
   case NONCON_READFAIL:
      printf("\nNon-Contiguous Read has failed\n");
      break;
   case NONCON_WRITEFAIL:
      printf("\nNon-Contiguous Write has failed\n");
      break;
   case NONCON_CONFFAL:
      printf("\nNon-Contiguous Read/Write configuration has failed\n");
      break;
   case FILL_FAIL:
      printf("\nFill Operation has failed\n");
      break;
   case READ_BLOCKFAIL:
      printf("\nContiguous Block Read has failed\n");
      break;
   case WRITE_BLOCKFAIL:
      printf("\nContiguous Block Write has failed\n");
      break;
   case SEARCH_FAIL:
      printf("\nSearch Tag Operation failed\n");
      break;
   case PROTECT_VIOLATE:
      printf("\nProtection violation\n");
      break;
   case NONCON_NOCONF:
      printf("\nNon-Contiguous Read/Write attempt without pre-configuration\n");
      break;
   case INP_NOTMATCH:
      printf("\nInput Command does not match pre-defined format\n");
      break;
   case COMM_FAIL:
      printf("\nCommunication time out.\n");
      break;
   default:
      printf("\nUnknown error #%d\n",error_num);
      break;  
   }
}

// ****************************************************************************
// Preform a Block Write.
//
// Input:        Start and stop tag addresses of a tag.
//               BOOLEAN - TRUE, using proteced mode block writes.
//                       - FALSE, normal block write.
// Return:       NONE.
// Side effects: NONE.
static void do_Write(WORD start_add, WORD end_add, BYTE *data_ptr, BYTE prt_mode)
{
   WORD now_start;  // Starting address for the current block write.
   WORD data_size;  // Amount of data to be written.
   WORD this_write; // Number of bytes in a given write block command.
   WORD data_index; // Arrary index for the data buffer pointer..
   BYTE status;     // Mifare tag interface status value.

   do
   {
      printf("\nSearching for tag.\n");
      start_down(TAG_START);  // This initialized finished.
      now_start = start_add;
      data_index = 0;         // Start with the first data input.
      data_size = end_add + 1 - start_add; // Count of tag data bytes.
      while (down_count() AND NOT finished) // Note: Because Tag IO,
      {          // mifare interface takes time the display down count is slower.
         if ((data_size / max_data_pkt) NE 0) // Next packet of data MAX size?
            this_write = max_data_pkt; // Use a maximum packet size for HMS827.
         else                          // else send just what fits.
            this_write = data_size % max_data_pkt;

         if((status = Write_command
            (now_start, this_write, data_ptr + data_index, prt_mode)) EQ OP_OK) 
         {
            printf("Writing to tag");     // No new line, just overprint same string.
            putch(CR);
            now_start = now_start + this_write; // Updata date address for next read.
            data_index = data_index + this_write; // Update data buffer array index.
            if ((data_size = data_size - this_write) EQ 0) // If no more data
               finished = TRUE;        // then, must be finished.
         } else if (status EQ PROTECT_VIOLATE)
         {
            print_code(status);
            display_delay();
            return;               // Write violation, give up.
         }
         if (esc_entered())
            return;               // Yes, get out. Exit on ESC.  Confused user.
      }
      if (finished)
         printf("Writing is done. \n");
      else
         printf("Operation timed out.\n");
   } while (ask_again()); // Repeat some command?
}

// ****************************************************************************
// Get (request) the data for a block write. Data entered through this routing
// is in number form not ASCII.  This is for Hexadecimal, Decimal, and Binary.
// The make a function call to do the write command.
//
// Input:        Start address to begin writing to a tag.
//               Last valid address because of prosection or tag size.
//               BOOLEAN - TRUE, using proteced mode block writes.
//                       - FALSE, normal block write.
// Return:       NONE.
// Side effects: NONE.
static void data_for_write(WORD start_add, WORD end_add, BYTE prt_mode)
{
   WORD data_size;  // Amount of data to be written.  Used as an array index.
   WORD now_addr;   // Current tag address a data byte is being entered for.
   WORD hold_num;   // Hold a WORD being input, for byte array.
   BYTE imp_char;   // Hold a character that is input. Like CR or ENTER.

   data_size = 0;   // Count of tag data bytes.
   now_addr = start_add;
   do
   {
      printf("%5u: ", now_addr);
      switch (display_type)
      {
      case HEX:
         hold_num = inp_hex();  // Enter a number in hexadecimal format.
         break;
      case DEC:
         hold_num = inp_num(BYTE_VALUE); // Enter a number in decimal format.
         break;
      case BIN:
         hold_num = inp_bin(); // Enter a number in binary format.
         break;
      default:                 // This will never happen.
         break;
      }
      if (hold_num EQ TOO_BIG_VAL)  // Was ESC entered?
         return;               // Yes, get out. Exit on ESC.  Confused user.
      if (hold_num NE ENTER_ALONE)
      {
         tag_io_buf[data_size] = hold_num;
         data_size = data_size + 1;  // Count one more byte to be written.
         if ((now_addr = now_addr + 1) > end_add)
         {
            printf("Last write address data was just entered.\n");
            printf("ENTER to begin the write, ESC to exit.\n"); 
            now_addr = now_addr - 1;  // No more data is to be entered.
            do
               if ((imp_char = getch()) EQ ESC)
                  return;            // The user is ESCaping out to quit.
            while (imp_char NE CR);
            hold_num = ENTER_ALONE;  // Exit the loop and preform the operation.
         }
      }
   } while (hold_num NE ENTER_ALONE);
   if (data_size > 0)                // If data entered, then do the write.
      do_Write(start_add, start_add + data_size - 1, tag_io_buf, prt_mode);
}

// ****************************************************************************
// Get (request) the data for a block write. This is to request an ASCII string.
// It is terminated with a carnage return.  Only enough characters are entered
// that will fit in the tag.  That is no more than string_length character entered.
// After this one should make a function call to do the write command.
//
// Input:        Total number of bytes allwed to be written from that address.
// Return:       Number of character entered into tag_io_buf[] for the write.
//               Return value of zero (0) is no bytes, like ENTER_ALONE.
//               if the user is ESCaping this function then TOO_BIG_VAL is returned.
// Side effects: tag_io_buf[] gets the input ASCII string.
static WORD get_ascii_data(WORD string_length)
{
   BYTE  inputting;           // BOOLEAN, TRUE while entering digits.
   BYTE  imp_char;            // Last character input.
   WORD  chr_pos;             // The string position (index) of the next character.
   WORD  maxed_pos;           // Number of character in the string, Maximum entered.
   
   maxed_pos = chr_pos = 0;   // No valid digits input yet.
   inputting = TRUE;
   memset(tag_io_buf, ' ', string_length); // Use SPACE as end of number.
   while (inputting)
   {
      if ((imp_char = getch()) EQ ESC)     // This echos the character.
         return(TOO_BIG_VAL);   // Use is trying to ESCape out.
      if (imp_char EQ ARROW_HI_BYTE)
      {
         if ((imp_char = getch()) EQ LEFT_ARROW)
         {
            if (chr_pos AND (chr_pos % DISPLAY_WIDTH))
            {  // Some digits have been input to back up over.
               putch(BACK_SPACE);            // Back up on the screen.
               chr_pos = chr_pos - 1;        // Backing up over a character.
            }
            continue;                     // Not an input character, but control.
         } else if (imp_char EQ RIGHT_ARROW)
         {
            if ((chr_pos <= string_length) AND (chr_pos < maxed_pos))
               putch(tag_io_buf[chr_pos++]); // Print again what is in the buffer.
            continue;                     // Not an input character, but control.
         }
      }
      if (imp_char EQ BACK_SPACE)
      {
         if (chr_pos AND (chr_pos % DISPLAY_WIDTH))
         {  // Some digits have been input to back up over.
            putch(imp_char);              // Echo what was just input.
            if (maxed_pos EQ chr_pos)     // GWP. BACK space deletes the character.
               maxed_pos = maxed_pos - 1; // GWP ??????????
            chr_pos = chr_pos - 1;        // Backing up over a character.
            tag_io_buf[chr_pos] = ' ';    // Last character is gone.
            putch(' ');                   // Erase from screen last character.
            putch(BACK_SPACE);            // Backup over the ' ' just printed.
         }
      } else if (imp_char EQ CR)
      {
         inputting = FALSE;         // Got a carriage return, thus done.
      } else if (chr_pos < string_length)// Overflowing?
      {                             // NO, can take another character.
         putch(imp_char);           // Echo what was just input.
         tag_io_buf[chr_pos++] = imp_char; // Got something, save it in input string.
         if (chr_pos > maxed_pos)
            maxed_pos = chr_pos;    // Sting getting longer.
      }
   }
   return(maxed_pos);
}

// ****************************************************************************
// Display data values in the current format.
//
// Input:        Pointer to data to be printed.
//               Number of bytes in the array pointed to.
//               Address in the tag of the data being printed. 
// Return:       BOOLEAN - TRUE if ESCAPE was entered when for next page display.
//                         FALSE ESCAPE was not entered, all data displayed.
// Side effects: None.
WORD display_data(BYTE* data_ptr, WORD num_bytes, WORD data_adr) 
{
   WORD line_number = 0; // nothing printed yet.
   
   while (num_bytes)    // Until all data bytes have been printed.
   {
      if (num_bytes >= format_size())
      {
         print_data_line(data_ptr, format_size(), data_adr);
         data_ptr = data_ptr + format_size();   // Point to next bytes to print.
         data_adr = data_adr + format_size();   // Tag address of bytes to print.
         num_bytes = num_bytes - format_size(); // Printed format_size() bytes.
      } else            // num_bytes must be < format_size
      {
         print_data_line(data_ptr, num_bytes, data_adr);
         num_bytes = 0; // Last line of bytes printed. Exit loop.
      }
      line_number = line_number + 1;  // Alother line printed.
      if ((line_number >= DISPLAY_LINES) AND num_bytes)
      {
         printf("Enter for next page: ");
         if (getche() EQ ESC)    // Use giving up and escaping out of this. 
            return (TRUE);       // ESCAPE entered!  EXIT!
         putch(CR);  // Be positive to be at the beginning of line.
         line_number = 0; // nothing printed on the new screen.
      }
   }   
   return(FALSE);       // Done displaying, no ESC entered.
}

// ****************************************************************************
// Print (display on standard out) an ASCII string of the current tag type
// that is expected to be attached to the reader.
// One leading space and enough spaces so all tag type strinngs are the same
// size.
//
// Input:        NONE.
// Return:       NONE.
// Side effects: Display updated. Output cursor moved..
void print_tag_type(void)
{
   switch (mem_size)
   {
   case HMS_MEM_SIZE:
      printf(" HMS ");
      break;
   default:
      printf("ERROR");
      break;
   }
}

#define SOME_DATA  5  // Try to read this many bytes for antenna and tag detect.
#define READ_SOMEPLACE 0x00 // Try to read from this address in antenna detect.
// ****************************************************************************
// Display if the current antenna type antenna is attached.
//
// Input:        Optional, comport to be used. 
// Return:       None.
// Side effects: None.
void ant_attached(void)
{
   BYTE ant_att;                // Local BOOLEAN flags for if antenna attatched
   BYTE tag_pres;               // and if a TAG is present.
   time_t start_time;           // Hold second count for display and tag read time.

   for ( ; ; ) // Loop forever, until user enters ESC.
   {
      ant_att = FALSE;          // Assume that no antenna with
      tag_pres = FALSE;         // Assume no tag present.
      // Search for a tag to see if antenna responds with anything.
      SearchTx(MIF_SEARCH_TI); // Try to communicate with the mifare chip.
      time(&start_time);
      while ((NOT comm_avail()) AND 
         (ANT_CHK_TIME > (time((long *)NULL) - start_time)))
      {
         if (esc_entered())       // Keep checking for an ESC key.
            return;               // Yes, get out. Exit on ESC.  Confused user.
      }
      if (comm_avail())         // Get any response from the antenna?
      {
         ant_att = TRUE;            // It is TRUE to have an antenna attached.
         if (SearchRx() EQ OP_OK)
            tag_pres = TRUE;        // TRUE to have a tag present.
      }
      clrscr();
      printf("Antenna for");
      print_tag_type();   // Print tag type for these commands.
      printf("\n");
      if (ant_att)
      {
         printf("    attached.\n"); // Yes, got something, it must be there. 
         if (tag_pres)
            printf("With Tag present.");
         printf("\n");
      } else
         printf("NOT attached.\n\n");
      time(&start_time);       // Get current second count form the runtime.
      do                       // Check for ESC entered, while waiting for antanna.
         if (esc_entered())
            return;    // ESC entered, exit this function.
      while (BETWEEN_CHECK > (time((long *)NULL) - start_time));
   }
}
      
// ****************************************************************************
// Continuously read 
// Infinite loop and increment block read 
//
// Input:        NONE.
// Return:       NONE.
// Side effects: NONE.
void ContinueRead(void) 
{
   WORD  data_size;  // Number of bytes for these reads.
   WORD  disp_start; // Tag starting address for the current read display line.
   WORD  now_start;  // Tag starting addres for the current block read.
   WORD  this_disp;  // Number of bytes in a given display line.
   WORD  left_disp;  // Number of bytes left to read for this_display.
   WORD  this_read;  // Number of bytes in a given read.
   BYTE *rx_data;    // Where to place the received tag data for a given read.
   WORD duration;    // Duration of continual read.
   time_t time_left; // Remaining time for this continual read.
   time_t now_last_time; // Value returned by time() last time.
   time_t old_last_time; // Hole the last time value for calculations.
   BYTE ii;        // Tempory varriable.         

   printf("\nDuration in seconds. Less then 32,767. ENTER for a minute:\n");
   if ((duration = inp_num(LARGEST_DEC)) EQ ENTER_ALONE)
      duration = 60;
   else if (duration EQ TOO_BIG_VAL)  // Was ESC entered? 
      return;     // Yes, get out. Exit on ESC.  Confused user.
   disp_start = 0;            // Begin at addres zero (0).
   data_size = mem_size;      // Display the entire tag..
   do                         // Do while the user keeps requesting more.
   {
      time_left = duration;
      time(&now_last_time);
      clrscr();
      printf("\nSearching for tag.\n");
      while (time_left > 0)               // For the duration entered to read.
      {
         old_last_time = now_last_time;   // Use as working varriable.
         time_left = time_left - (time(&now_last_time) - old_last_time);
         // The time left value used to be printed out, so it is calculated.
         if ((data_size / format_size()) NE 0)  // Next packet of data MAX size?
            this_disp = format_size(); // Use a maximum display for this format..
         else                          // else just what fits.
            this_disp = data_size;
            
         rx_data   = tag_io_buf;       // Use the global working tag data buffer.
         now_start = disp_start;       // Line of data starts at this tag address.
         left_disp = this_disp;        // Have read no bytes for this display.
         start_down(TAG_CONTINUAL);    // This initialized finished.
         do                            // Do reads for one line of display.
         {
            while (down_count() AND NOT finished) // Note: Because BLReadTx and
            {            // BLReadRx take time the display down count is slower.
               if (left_disp <= max_data_pkt) // One display line be read with
                  this_read = left_disp;      // one read command.
               else                           // OR, 
                  this_read = max_data_pkt;   // many reads at maximum size.
               if (Read_command(now_start, this_read, rx_data) EQ OP_OK) 
               {
                  rx_data = rx_data + this_read;  // For next read of this display.
                  now_start = now_start + this_read;// Got this_read more tag data.
                  if ((left_disp = left_disp - this_read) EQ 0)// Read bytes.
                     finished = TRUE;  // This display bytes have been read.
               }
               if (kbhit())
               {
                  if ((ii = getch()) EQ ESC)
                     return;
                  if (ii EQ CR)
                  {
                     printf("\nENTER to repeat.\n");
                     do
                     {
                        if ((ii = getch()) EQ ESC)
                           return;             // User is giving up.
                     } while (ii NE CR);    // CR is ENTER to repeate.
                     printf("\nSearching for tag.\n");
                     time_left = duration;  // Start fresh display time.
                     time(&now_last_time);
                  }
               }
            }
         }  while (left_disp AND down_count());
         if (finished)
         {     //print out receiving data
            display_data(tag_io_buf, this_disp, disp_start);
            data_size = data_size - this_disp;
            if ((disp_start = disp_start + this_disp) >= mem_size)
            {
               disp_start = 0;  // Loop back to the begining.
               data_size = mem_size;      // Display the entire tag..
            }
         } 
      }
   } while (ask_again()); // Repeat some command?
}

// ****************************************************************************
// Preform a Non-contiguous write. The address must have been setup in the last
// command, using read_write_conf().
//   
// Input:        NONE.
// Return:       NONE.
// Side effects: NONE.
void write_notcont(void) 
{
   BYTE status;                // Status of write command.
   WORD num_write;             // Number of write  address input.
   WORD num_ASCII;             // Number of ASCII data bytes entere on a given line.
   BYTE data_buf[OUT_LIMIT];   // build the command in this buffer.

   num_write = 0;
   address_ptr = data_buf;
   do
   {
      printf("Input data byte number %d (ENTER when finished): ", num_write + 1);
      switch (display_type)
      {
      case HEX:
         address_value = inp_hex();  // Enter a number in hexadecimal format.
         break;

      case DEC:
         address_value = inp_num(BYTE_VALUE); // Enter a number in decimal format.
         break;

      case BIN:
         address_value = inp_bin(); // Enter a number in binary format.
         break;

      case ASCII:
         num_ASCII = get_ascii_data(1);    // Get a byte of ASCII data, a character.
         if (num_ASCII EQ TOO_BIG_VAL)
            address_value = TOO_BIG_VAL;   // User trying to ESCape out of this command.
         else if (num_ASCII EQ 0)
            address_value = ENTER_ALONE;   // Done entering the characters.
         else
            address_value = tag_io_buf[0]; // from a common buffer (return parameter).
         printf("\n");                     // Output a carriage return.
         break;

      default:                 // This will never happen.
         address_value = TOO_BIG_VAL; // ERROR, this should never happen.
         break;
      }
      if (address_value EQ TOO_BIG_VAL)
         return;     // User trying to ESCape out of this command.
      if (address_value NE ENTER_ALONE)
         num_write++;        // Data for another address was input, count it.
      *address_ptr++ = (BYTE) address_value & 0xFF; // and low byte also.
   } while (address_value NE ENTER_ALONE);                   // And point past data.

   // Prefrom the Non-contiguous write, with write time + some more for big write commands.
   NCWriteTx(MIF_WRITE_TI + (num_write * 10), data_buf, num_write);
   status = NCWriteRx();
   if(status EQ OP_OK) 
      printf("\nNon-contiguous write is done. \n");
   else      //error code is RX
      print_code(status);
   printf("\nPress any key to go back to menu. \n");
   getch();
}

// ****************************************************************************
// Preform a Non-contiguous read. The address must have been setup in the last
// command, using read_write_conf().
//   
// Input:        NONE.
// Return:       NONE.
// Side effects: NONE.
void read_notcont(void) 
{
   WORD ii;
   BYTE this_fmt_size;  // How many bytes displayed on a line for this formt.
   BYTE *rx_buf; 
   BYTE status;

   printf("\nReading data from tag.  Please wait...\n\n");
   NCReadTx(MIFARE_TIME_OUT);
   
   rx_buf = (BYTE*) malloc(nc_read_size);
   status = NCReadRx(rx_buf, nc_read_size);

   printf("\n");
   if( (status EQ OP_OK) ) 
   {     //print out receiving data.
      if (display_type EQ ASCII)  // If displaying in ASCII, then
         this_fmt_size = format_size() / 2; // This ASCII display has a space between
      else                                  // that characters. Non-contigous data.
         this_fmt_size = format_size();
      for(ii = 0; ii < nc_read_size; ii++)
      {
         print_byte(rx_buf[ii]);
         if (display_type EQ ASCII)  // If displaying in ASCII, then
            printf(" ");  // have a space between characters.
         if (((ii + 1) MOD this_fmt_size) EQ 0) // If handled a display line of data,
            printf("\n");                 // then output a carriage return.
      }
   } else                     //receive error message
      print_code(status);

   free(rx_buf);
   printf("\nPress any key to go back to menu. \n");
   getch();

}

// ****************************************************************************
// Set up for a Read/Write Configuration Command.  This is to configure the
// tag for non-Contiguour Read/Write 
//   
// Input:        NONE.
// Return:       NONE.
// Side effects: NONE.
void read_write_conf(void) 
{
   BYTE *write_addr;    // Point to the input write addresses.
   BYTE status;  // Status of write command.
   WORD num_write;      // Number of write  address input.

   address_ptr = tag_io_buf;
   nc_read_size = 0;  // To remember How many bytes read. 
   do
   {
      printf("Input read addresses number %d (ENTER when finished): ", nc_read_size + 1);
      if ((address_value = inp_num(mem_size)) EQ TOO_BIG_VAL)
         return;     // User trying to ESCape out of this command.
      if (address_value NE ENTER_ALONE)
         nc_read_size++;        // Another address  was input, count it.
      *address_ptr++ = address_value >> 8;   // Get the high byte
      *address_ptr++ = (BYTE) address_value & 0xFF; // and low byte also.
   } while (address_value NE ENTER_ALONE);                   // And point past data.
   write_addr = address_ptr;
   num_write = 0;
   do
   {
      printf("Input write addresses number %d (ENTER when finished): ", num_write + 1);
      if ((address_value = inp_num(mem_size)) EQ TOO_BIG_VAL)
         return;     // User trying to ESCape out of this command.
      if (address_value NE ENTER_ALONE)
      {
         num_write++;
         *address_ptr++ = address_value >> 8;   // Get the high byte
         *address_ptr++ = (BYTE) address_value & 0xFF; // and low byte also.
      }
   } while (address_value NE ENTER_ALONE);
   printf("Setting up for non-contiguous I/O. Please wait...");
   NCConfTx(MIFARE_TIME_OUT, tag_io_buf, nc_read_size, write_addr, num_write);
   status = NCConfRx();
   if(status EQ OP_OK) 
      printf("\nNon-contiguous configuration is done. \n");
   else      //error code is RX
      printf("\nConfiguration Command fails with Error Code [%02X].\n", status);
   printf("\n\nHit  ANY  key to continue ... ");
   getch();
}

// ****************************************************************************
// Prefrom a block read.
//
// Input:        NONE.
// Return:       NONE.
// Side effects: NONE.
void BlockRead(void) 
{
   WORD  start_add;  // Starting address for these block read.
   WORD  end_add;    // Ending address for these block read.
   WORD  data_size;  // Number of bytes for these reads.
   WORD  disp_start; // Tag starting address for the current read display line.
   WORD  now_start;  // Tag starting addres for the current block read.
   WORD  this_disp;  // Number of bytes in a given display line.
   WORD  left_disp;  // Number of bytes left to read for this_display.
   WORD  this_read;  // Number of bytes in a given read.
   BYTE *rx_data;    // Where to place the received tag data for a given read.

   clrscr();
   if (NOT req_tag_range(&start_add, &end_add, 0, mem_size - 1)) // Get tag data bounds.
      return;                // Not valid range, give up and get nothing.
   do                        // The entire tag can be read from.
   {
      disp_start = start_add;
      data_size = end_add + 1 - start_add;      // Count of tag data bytes.
      printf("\nSearching for tag.\n");
      while (data_size)      // While more data to be displayed.
      {
         if ((data_size / format_size()) NE 0)  // Next packet of data MAX size?
            this_disp = format_size(); // Use a maximum display for this format..
         else                          // else send just what fits.
            this_disp = data_size;
            
         rx_data   = tag_io_buf;       // Use the global working tag data buffer.
         now_start = disp_start;       // Line of data starts at this tag address.
         left_disp = this_disp;        // Have read no bytes for this display.
         start_down(TAG_START);        // This initialized finished.
         do
         {
            if (left_disp <= max_data_pkt) // One display line be read with
               this_read = left_disp;      // one read command.
            else                           // OR, 
               this_read = max_data_pkt;   // many reads at maximum size.
            finished = FALSE;              // Not finished, more byte to read.
            while (down_count() AND NOT finished) // Note: Because BLReadTx and
            {            // BLReadRx take time the display down count is slower.
               if (Read_command(now_start, this_read, rx_data) EQ OP_OK) 
               {
                  rx_data = rx_data + this_read;  // For next read of this display.
                  left_disp = left_disp - this_read;// Read bytes.
                  now_start = now_start + this_read;// Got this_read more tag data.
                  finished = TRUE;  // This display bytes have been read.
               }
               if (esc_entered())
                     return;                 // User is giving up.
            }
         }  while ((left_disp) AND down_count());
         if (finished)
            {     //print out receiving data
               display_data(tag_io_buf, this_disp, disp_start);
               data_size = data_size - this_disp;
               disp_start = disp_start + this_disp; // Updata date address for next read.
            } 
         else
            {
               printf("\nOperation timed out.\n");
               data_size = 0;   // Read error. Get out.
            }
      }
   } while (ask_again()); // Repeat some command?
}

// ****************************************************************************
// Preform a protected block write command.
//
// Input:        BOOLEAN - TRUE, using proteced mode block writes.
//                       - FALSE, normal block write.
// Return:       NONE.
// Side effects: NONE.
void pro_BlockWrite(BYTE prt_mode)
{
   WORD start_add;  // Starting address for the write.
   WORD end_add;    // Ending address for these block write.
   WORD valid_start;// this tag's starting address, if USE or USE or protected.

   clrscr();
   valid_start = HMS_TAG_START;       // Get the starting tag address.
   printf("Starting address, (%u to %u): ", valid_start, mem_size - 1);
   if ((start_add = inp_tag_range(valid_start, mem_size - 1)) EQ TOO_BIG_VAL)
      return;                // User is ESCaping from this command.
   if (start_add EQ ENTER_ALONE) // No digits entered. Start at 
   {
      start_add = valid_start;   // the lowest valid tag address. 
      printf("Start address %u\n", start_add);
   }
   if (display_type EQ ASCII)
   {
      printf("Input string:\n");
      end_add = get_ascii_data(mem_size - start_add);
      if ((end_add EQ 0) OR (end_add EQ TOO_BIG_VAL))// User ESCaping.  
         return;  // Or entered no data bytes (characters) so, EXIT this command.
      end_add = start_add + (end_add - 1); // Figure the ending address.
      do_Write(start_add, end_add, tag_io_buf, prt_mode);
   }
   else // It is not an ASCII string. So enter numbers individually.
      data_for_write(start_add, mem_size - 1, prt_mode); // Request data, preform write.
}

// ****************************************************************************
// Preform a get tag's Identification number command?
//
// Input:        NONE.
// Return:       NONE.
// Side effects: NONE.
void Get_tag_ID(void) 
{
   BYTE finished;        // BOOLEAN: Local BOOLEAN Not finished yet command.
   unsigned long int ID_number;  // Hold to be printed tag's Identification number.
   WORD LptCnt;          // Little local loop counter.
   BYTE status;                // Status of write command.

   do
   {
      printf("\nSearching for tag.\n");
      start_down(TAG_START);  // Setup for timing down count.
      finished = FALSE;       // Not finished with this operation yet.
      while (down_count() AND NOT finished) // Note: Because Tag IO,
      {          // mifare interface takes time the display down count is slower.
         Get_Tag_IDTx(MIF_SEARCH_TI);
         
         if ((status = Get_Tag_IDRx(tag_io_buf)) EQ OP_OK) 
            finished = TRUE;
         else if ((status NE SEARCH_FAIL) AND (status NE COMM_FAIL))
            print_code(status);
         if ((status EQ INP_NOTMATCH) OR esc_entered())
         {
            printf("\nPress any key to go back to menu. \n");
            getch();
            return;                     // User is giving up.
         }
      }
      if (finished)
      {
         ID_number = (unsigned long int) tag_io_buf[0];
         for (LptCnt = 1; LptCnt < ID_NUM_SIZE; LptCnt++)
            ID_number = (ID_number << 8) + (unsigned long int) tag_io_buf[LptCnt];
         printf("\nIn decimal, the identification number is: %lu\n", ID_number);
         printf("In hexadecimal the bytes are: 0x%02X%02X%02X%02X\n", 
            tag_io_buf[0], tag_io_buf[1], tag_io_buf[2], tag_io_buf[3]);
      } else
         printf("\nSearch timed out.  Identification number not found.\n");
   } while (ask_again()); // Repeat some command?
}

// ****************************************************************************
// Preform a search tag. Is a tag present?
//
// Input:        NONE.
// Return:       NONE.
// Side effects: NONE.
void SearchTag(void) 
{
   BYTE status;

   do
   {
      printf("\nSearching for tag.\n");
      start_down(TAG_START);  // This initialized finished.
      while (down_count() AND NOT finished) // Note: Because Tag IO,
      {          // mifare interface takes time the display down count is slower.
         SearchTx(MIF_SEARCH_TI);
         status = SearchRx();
         if( status EQ OP_OK) 
            finished = TRUE;
         if (esc_entered())
            return;                     // User is giving up.
      }
      if (finished)
         printf("\nTag is found.\n");
      else
         printf("\nSearch timed out.\n");
   } while (ask_again()); // Repeat some command?
}

// *****************************************************************************
// Preform a clear tag function.
//
// Input:        NONE.
// Return:       NONE.
// Side effects: NONE.
void ClearTag(void) 
{
   BYTE status;

   printf("Clearing tag!\nENTER to continue:");
   if (getche() NE CR)
      return;
   do
   {
      printf("\n");
      start_down(TAG_START);  // This initialized finished.
      while (down_count() AND NOT finished) // Note: Because Tag IO,
      {          // mifare interface takes time the display down count is slower.
         printf("Searching for tag.");
         putch(CR);
         do
         {
            SearchTx(MIFARE_TIME_OUT);
            status = SearchRx();
            if (esc_entered())
               return;                     // User is giving up.
         } while ((status NE OP_OK) AND down_count());
         if (status EQ OP_OK)
         {
            printf("Clearing tag.     ");
            putch(CR);
            FillTx(0, 0, MIFARE_CLR_SET, 0);
            status = FillRx();
            if(status EQ OP_OK) 
               finished = TRUE;
         }
         if (esc_entered())
            return;                     // User is giving up.
      }
      if(status EQ OP_OK) 
         printf("\n\nTag cleared to zero\n");
      else   //error code is RX
         printf("\n\nClear not successful\n");
   } while (ask_again()); // Repeat some command?
}

// ****************************************************************************
// Preform a fill tag with some value command.
//
// Input:        NONE.
// Return:       NONE.
// Side effects: NONE.
void FillTag(void) 
{
   BYTE status;    // Tag IO return status.
   BYTE data;      // The data to be written (filled) to the tag.
   WORD value;     // Value (or ESC flag) entered by user to be filled.
   WORD start_add; // The starting address of where the data will be filled.
   WORD end_add;   // The ending address of the data to be filled.
   WORD data_size; // Number of bytes to fill with the data entered.
   WORD valid_start;// this tag's starting address, if USE or USE or protected.

   clrscr();
   valid_start = HMS_TAG_START;       // Get the starting tag address.
   printf("Fill tag operation.\n");
   // Get starting address and data size to read
   if (NOT req_tag_range(&start_add, &end_add, valid_start, mem_size - 1)) // Get tag data bounds.
      return;                // Not valid range, give up and get nothing.
   data_size = end_add + 1 - start_add;  // Count of tag data bytes.

   do
   {
      if (display_type EQ ASCII)
      {
         printf("Enter an ASCII character to fill: ");
         do
         {
            if ((status = getche()) EQ ESC) // This echos the character.
               return;                      // But ESC was entered. Exit.
            if (status EQ BACK_SPACE)
            {
               putch(' ');                  // Erase from screen last character.
               value = ' ';                 // Back space erased the character.
               putch(BACK_SPACE);           // Backup over the ' ' just printed.
            } else if (status NE CR)
               value = status;              // Got a character input.
            putch(BACK_SPACE);              // Stay in first column, for 1 char.
         } while (status NE CR);
      } else if (display_type EQ HEX)
      {
         printf("Enter the data in Hexadecimal: ");
         value = inp_hex(); // Enter a number in hexadecimal format.
      } else if (display_type EQ BIN)
      {
         printf("Enter the data in Binary: ");
         value = inp_bin(); // Enter a number in hexadecimal format.
      } else
      {
         printf("Enter the data in decimal [0..255]: ");
         value = inp_num(BYTE_VALUE);
      }
      if (value EQ TOO_BIG_VAL)  // Was ESC entered?
         return;      // Yes, get out. Exit on ESC.  Confused user.
      if (value EQ ENTER_ALONE)
      {
         clrscr();
         printf("\nEnter fill value for %u through %u.\n\n", start_add, end_add);
      }
   } while (value EQ ENTER_ALONE); 
   data = (BYTE) value;        // Cast the value WORD to a data BYTE.

   do
   {
      printf("\n");           // Extra new line after entered number.
      start_down(TAG_START);  // This initialized finished.
      while (down_count() AND NOT finished) // Note: Because Tag IO,
      {          // mifare interface takes time the display down count is slower.
         printf("Searching for tag.");
         putch(CR);
         do
         {
            SearchTx(MIFARE_TIME_OUT);
            status = SearchRx();
            if (esc_entered())
               return;                     // User is giving up.
         } while ((status NE OP_OK) AND down_count());

         if (status EQ OP_OK)
         {
            printf("Filling tag.      ");
            putch(CR);
            FillTx(start_add, data_size, MIFARE_CLR_SET, data);
            status = FillRx();
            if (status EQ OP_OK) 
               finished = TRUE;
         }
         if (esc_entered())
            return;                     // User is giving up.
      }
      if (status EQ OP_OK)
         printf("\n\nFill tag is done.\n");
      else      //error code is RX
         printf("\n\nFill not successful\n");
   } while (ask_again()); // Repeat same command?
}

// ****************************************************************************
// Preform a copy from tag command.  The tag value is saved in tag_buf[].
//
// Input:        NONE.
// Return:       NONE.
// Side effects: tag_buf[] is updated, with range array indexes.
void copy_from_tag(void)
{
   WORD cur_adr;     // Currently working with this tag address. 
   WORD curnt_blks;  // Data size for current write command.
   BYTE status;      // Status of read command.
   int  error_count; // Count retry errors, not set correctly.
   WORD begin_tag;   // Begin coping from this address.
   WORD end_tag;     // Last address byte to be coping.
   WORD valid_start; // this tag's starting address, if USE or USE or protected.
   BYTE retrying;    // BOOLEAN - TRUE while retrying to read from a tag.
                     //           FALSE got desired data.

   valid_start = HMS_TAG_START;   // HMS tags can write to address zero.
   if (NOT req_tag_range(&begin_tag, &end_tag, valid_start, mem_size - 1)) // Get tag data bounds.
      return;                // Not valid range, give up and get nothing.
   memset(tag_buf, BLANK_STR, mem_size);       // Assume all values not valid.
   start_adr = TOO_BIG_VAL;  // 'Flag' to indicate no tag data has been read.
   stop_adr  = TOO_BIG_VAL;
   do
   {
      if ((end_tag - begin_tag) >= max_data_pkt)
         curnt_blks = max_data_pkt;
      else
         curnt_blks = end_tag - begin_tag + 1;  // Total size.


      error_count = 0;         // Start counting errors again.
      retrying = FALSE;        // Start fresh again.
      cur_adr = begin_tag;
      printf("\nSearching for tag.\n");
      while (cur_adr <= end_tag)  // For all max packet blocks.
      {
         if ((status = Read_command(cur_adr, curnt_blks, &tag_buf[cur_adr])) EQ OP_OK) 
         {
            printf("Read from %d", cur_adr + curnt_blks - 1);
            putch(CR);
            cur_adr = cur_adr + curnt_blks;
            if ((cur_adr + curnt_blks) > end_tag)
               curnt_blks = end_tag - cur_adr + 1;
         } else if (error_count++ > TOO_MANY_ERRORS)
         {   // That read did not work, try again?
            clrscr();
            print_code(status);     // Print Error Code status return message.
            printf("\nNot all read\n");
            printf("Enter to try again\n");
            if (getche() EQ ESC)    // Use giving up and escaping out of this. 
               return;
            retrying = TRUE;         // Try again.
            break;
         }
         if (esc_entered())
            return;                     // User is giving up.
      }
   } while (retrying);

   start_adr = begin_tag;   // New tag values have been read in.
   stop_adr  = end_tag;     // So, the global array indexes are update.
   
#if FALSE   
for (error_count = start_adr; error_count <= stop_adr; error_count++)
{
   if (error_count % 4 EQ 0)
      printf("\n %3d:  ", error_count);
   printf("%02X ", tag_buf[error_count]);
}
#endif

   printf("\nTag data read.\n");
   display_delay();
}

// ****************************************************************************
// Preform a copy to a tag command.  The tag data is gotten from what was read
// during a tag copy.
//
// Input:        NONE.
// Return:       NONE.
// Side effects: NONE.
void copy_to_tag(void)
{
   WORD now_adr;     // Currently working with this tag address. 
   WORD curnt_blks;  // Data size for current write command.
   BYTE status;      // Status of write command.
   int  error_count = 0; // Count retry errors, not set correctly.
 
   if ((start_adr NE TOO_BIG_VAL) AND (stop_adr NE TOO_BIG_VAL))
   {  // Only write a saved tag and range.
      if ((stop_adr - start_adr) >= max_data_pkt)
         curnt_blks = max_data_pkt;
      else
         curnt_blks = stop_adr - start_adr + 1;  // Total size.
      now_adr = start_adr;   // Now, begin at the start of valid data.
      
      printf("\nSearching for tag.\n");
      while (now_adr <= stop_adr)  // For all max packet blocks.
      {
         if ((status = Write_command(now_adr, curnt_blks, &tag_buf[now_adr], FALSE))
            EQ OP_OK)
         {  // Sucessful write, update pointers for the next write.
            printf("Wrote to %d", now_adr + curnt_blks - 1);
            putch(CR);
            now_adr = now_adr + curnt_blks;  
            if ((now_adr + curnt_blks) > stop_adr)
               curnt_blks = stop_adr - now_adr + 1;
         } else
         {  // That read did not work, try again.
            if (error_count++ > TOO_MANY_ERRORS)
            {
               clrscr();
               print_code(status);     // Print Error Code status return message.
               printf("\nNot all written\n");
               printf("ENTER for menu.\n");
               getche();
               return;         // Test failed. Did not write to the entire tag.
            }
         }
         if (esc_entered())
            return;                     // User is giving up.
      }
   }
   printf("\nTag data written.\n");
   display_delay();
}

// ****************************************************************************
// Print for viewing the data coped into tag_buf[], using an above sub routine.
//
// Input:        NONE.
// Return:       NONE.
// Side effects: NONE.
void disp_copied(void)
{
   if ((start_adr NE TOO_BIG_VAL) AND (stop_adr NE TOO_BIG_VAL))
   {  // Only write a saved tag and range.
      do
      {
         if (display_data(&tag_buf[start_adr], (stop_adr - start_adr + 1), start_adr))
            return;          // Escape entere, thus exit.
      } while (ask_again()); // Repeat same command?
   } else
   {
      printf("No data to be displayed\n");
      display_delay();
   }
}

