// ****************************************************************************
//
// File Name:    HMS800.c
//
// Description: Preform the HMS commands from here.  Calls are made to
//              routines in comm.c
//
//  ORIGINATOR: Unknown.
//
//  HISTORY
//    who     when     what
// -------- --------- ----------------------------------------------------
//  Geof    08-??-97  Clean up, add marker line at top of every routine.
//  Geof    10-12-97  Extend timeout.  Wait for the device, for HMS814.
//                    Added debug statments for Non-contiguous IO.
//  Geof    01-12-98  Took out calls to malloc and used a static array.  This
//                    was done for timing analysis, less time overhead.
//  Geof    03-24-98  Added now_mux_addr when using MUX32 format. This is so
//                    the MUX address can be changed.
//  Geof    04-06-98  Moved om_buf_clr(), com_rec_buf(), and SendStream() 
//                    from hms800.c to this comm.c.
//  Geof    06-05-98  Moved now_mux_addr declaration to comm.c.
//  Geof    11-13-98  Clean up ABx response (error or OK) must be received.
//  Geof    02-09-99  Added the Get tag Identification Number command.
//
// ****************************************************************************

#include <stdlib.h>
#include <string.h>
#include <dos.h>
#include "typedef.h"
#include "handheld.h"
#include "hms800.h"
#include "comm.h"

#define DEBUGGING  FALSE
#if DEBUGGING
#include <stdio.h>
#endif

#if ANTARES
#include "imt24lib.h"  // Intermet Applicatoin Development tools.w
#endif

#define MORE_THAN_ENOUGH  (256 * 2)
#define REC_FIRST_BYTE    (MIFARE_TIME_OUT >= 1000 ? ((MIFARE_TIME_OUT / 1000) + 2) : 2)

#define REC_MORE_BYTES      5  // Receiving about 4 bytes.
#define REC_BYTES_TIME      2  // Seconds waiting for bytes in a packet.

// Below working variables for non-contiguous I/O routines.
BYTE build_buf[MORE_THAN_ENOUGH];

// ****************************************************************************
// Clear a buffer 
// Input:        Pointer to a buffer to be cleared.
//               Number of bytes in the buffer, to be clared.
// Return:       None.
// Side effects: None.
static void ClearBuf (BYTE *rx_buf, WORD data_size) 
{
   memset(rx_buf, 0x00, data_size);   // Assume all values not valid.
}

// ****************************************************************************
// Transmit Block Write Command 
// Input:        Begin write at this start address in the tag.
//               Number of bytes to be written from tag.
//               Antenna timeout time.
//               Pointer to the bytes to be written.
//               The block write mode, that is normal or protected.
// Return:       None.
// Side effects: None.
static void do_BLWrite(WORD start_add, WORD data_size, WORD timeout, BYTE *data_buf, BYTE mode)
{
    BYTE *tx_buf;
    WORD ii;

    //allocate buffer and initialize them
    tx_buf = build_buf;
    ClearBuf(tx_buf, data_size*2 + 10);

    *tx_buf++ = MSG_BEGIN;
    *tx_buf++ = mode;
    *tx_buf++ = (BYTE)(start_add / 256);
    *tx_buf++ = (BYTE)(start_add % 256);
    *tx_buf++ = (BYTE)(data_size / 256);
    *tx_buf++ = (BYTE)(data_size % 256);
    *tx_buf++ = (BYTE)(timeout / 256);
    *tx_buf++ = (BYTE)(timeout % 256);
    for(ii = 0; ii < data_size; ii++) 
    {
        tx_buf++;
        *tx_buf++ = *data_buf++;
    }
    *tx_buf++ = MSG_TERM;
    *tx_buf = MSG_TERM;
   
#if ANTARES
    com_buf_clr();              // Clear out Anyares COM receive buffer.
#endif    
    SendStream(build_buf, data_size*2 + 10);
}

// ****************************************************************************
// Receive response from block read.
// Input:        where to place the data.
//               Receive no more data than this. Note: BYTE received in a WORD.
//               The read mode, Block read or Continual, BLK_READ or CON_READ_MODE.
// Return:       Command status.
// Side effects: None.
static BYTE do_BLReadRx(BYTE *data_buf, WORD data_size, BYTE read_mode)
{
   BYTE *rx_buf;
   BYTE status;
   WORD ii;

    //allocate buffer and initialize them
   rx_buf = build_buf;
   ClearBuf(rx_buf, data_size*2 + 4);

#if TEST_MUX32
   if(NOT com_rec_buf(rx_buf, 2, REC_FIRST_BYTE)) // Throw out the first two bytes. 
      status = COMM_FAIL;   // They are MUX32 address and stuff.
   else
#endif  
   // first check the first 2 bytes to find out what is expected
   // if even the first 2 bytes are not gotten, serial communication fails
   // if get 0xff in 2nd bytes, receive Error message
   // else may get right message
   if (NOT com_rec_buf(rx_buf, 2, REC_FIRST_BYTE))
       status = COMM_FAIL;
   else if (rx_buf[MSG_FUN_INX] EQ 0xff) 
   {
       if (NOT com_rec_buf(rx_buf + 2, 4, REC_BYTES_TIME))
           status = COMM_ER1;
       else 
           status = rx_buf[3];  // Get status returned form antenna.
   } else 
   {
       if (com_rec_buf(rx_buf + 2, data_size*2 + 2, REC_MORE_BYTES)
           AND (rx_buf[0] EQ MSG_BEGIN) AND (rx_buf[MSG_FUN_INX] EQ read_mode))
       {
           status = OP_OK;
           for(ii=0; ii < data_size; ii++) 
               *data_buf++ = rx_buf[3 + ii*2];
       } else 
           status = COMM_INV;
   }

   return(status);
}

// ****************************************************************************
// If the received message is short one (4 bytes) without data, use this one
// Input:        The command number that is being echoed as received.
// Return:       BOOLEAN: TRUE  - got expected command echo.
//                        FALSE - did not get expected command echo.
// Side effects: None.
static BYTE RxCheck(BYTE command_no)
{
    BYTE  rx_buf[6], status;

    ClearBuf(rx_buf, 6);

#if TEST_MUX32
    if(NOT com_rec_buf(rx_buf, 2, REC_FIRST_BYTE)) // Throw out the first two bytes. 
       status = COMM_FAIL;   // They are MUX32 address and stuff.
   else
#endif
   // first check the first 2 bytes to find out what is expected
   // if even the first 2 bytes are not gotten, serial communication fails
   // if get 0xFF in 2nd bytes, receive Error message
   // else may get right message
   if (NOT com_rec_buf(rx_buf, 2, REC_FIRST_BYTE))
   {
       status = COMM_FAIL;
   } else if (rx_buf[MSG_FUN_INX] EQ 0xff) 
   {
       if (com_rec_buf(rx_buf+2, 4, REC_BYTES_TIME))
           status = rx_buf[3];
       else
           status = COMM_FAIL;
   } else 
   {
   if (com_rec_buf(rx_buf+2, 2, REC_MORE_BYTES) AND 
       (rx_buf[MSG_BEG_INX] EQ MSG_BEGIN) AND (rx_buf[MSG_FUN_INX] EQ command_no)
       AND (rx_buf[2] EQ MSG_TERM) AND (rx_buf[3] EQ MSG_TERM) ) 
           status = OP_OK;
       else
           status = COMM_INV;
   }
   return(status);
}

           
// ****************************************************************************
// Non-Contiguour Read Command 
//   
// Input:        Time out value of some type.  It is sent in the command packet.
// Return:       None.
// Side effects: A packet is queued to be sent.
void NCReadTx(WORD timeout) 
{
    BYTE tx_buf[6];
    
    tx_buf[MSG_BEG_INX] = MSG_BEGIN;
    tx_buf[MSG_FUN_INX] = NONCO_READ;
    tx_buf[MSG_TIMH_IX] = (BYTE)(timeout / 256);
    tx_buf[MSG_TIML_IX] = (BYTE)(timeout % 256);
    tx_buf[4] = MSG_TERM;
    tx_buf[5] = MSG_TERM;

#if ANTARES
    com_buf_clr();              // Clear out Anyares COM receive buffer.
#endif    
    SendStream(tx_buf, 6);
}
     
     
// ****************************************************************************
// Transmit the Non-Contiguour Write Command 
//   
// Input:        Time out value of some type.  It is sent in the command packet.
//               Pointer to the data to be written.
//               Number of data bytes.
// Return:       NONE.
// Side effects: A packet is queued to be sent.
void NCWriteTx(WORD timeout, BYTE *data_buf, 
                      WORD data_size)
{
    WORD i;
    BYTE *address_ptr; // Index the output buffer to build command packet.

    ClearBuf(build_buf, (data_size * 2) + 6);  // initialize buffer.

    build_buf[MSG_BEG_INX] = MSG_BEGIN;
    build_buf[MSG_FUN_INX] = NONCO_WRITE;
    build_buf[MSG_TIMH_IX] = (BYTE)(timeout / 256);
    build_buf[MSG_TIML_IX] = (BYTE)(timeout % 256);
    address_ptr = &build_buf[4];         // Setup for loop index pointer.
    for(i = 0; i < data_size; i++)
    {
        address_ptr++;
        *address_ptr++ = *data_buf++;
    }
    *address_ptr++ = MSG_TERM;
    *address_ptr++ = MSG_TERM;

#if ANTARES
    com_buf_clr();              // Clear out Anyares COM receive buffer.
#endif    
    SendStream(build_buf, (data_size * 2) + 6);
}
  
// ****************************************************************************
// Non-Contiguous Read/Write Configuration Command 
// Input:        Time out value of some type.  It is sent in the command packet.
//               Pointer to two buffers, that go in the read/write command.
//               Number of bytes in each buffer.
// Return:       NONE.
// Side effects: A packet is queued to be sent.
void NCConfTx(WORD timeout, BYTE *read_buf, WORD read_size, BYTE *write_buf,
              WORD write_size) 
{
    BYTE *tx_buf;
    WORD buf_size, ii;
    
    buf_size = read_size * 2 + write_size * 2 + 10; 
    tx_buf = build_buf;
    ClearBuf(tx_buf, buf_size);   // initialize buffer.

    *tx_buf++ = MSG_BEGIN;
    *tx_buf++ = CONF_NONCO;
    *tx_buf++ = (BYTE)(timeout / 256);
    *tx_buf++ = (BYTE)(timeout % 256);
    for(ii =0; ii < read_size; ii++)
    { 
        *tx_buf++ = *read_buf++;    // The address WORD
        *tx_buf++ = *read_buf++;    // has two bytes.
    }
    *tx_buf++ = 0xff;
    *tx_buf++ = 0xf0;
    for(ii = 0; ii < write_size; ii++) 
    { 
        *tx_buf++ = *write_buf++;   // The address WORD
        *tx_buf++ = *write_buf++;   // has two bytes.
    }
    *tx_buf++ = 0xff;
    *tx_buf++ = 0xf1;
    *tx_buf++ = MSG_TERM;
    *tx_buf++ = MSG_TERM;

#if ANTARES
    com_buf_clr();              // Clear out Anyares COM receive buffer.
#endif    
    SendStream(build_buf, buf_size);
}

// ****************************************************************************
// Transmit the Fill Command 
// Input:        Begin fill at what address in the tag?
//               How many bytes to fill with the data byte.
//               Antenna timeout time.
//               The data to be filled in the tag.
// Return:       None.
// Side effects: None.
void FillTx(WORD start_add, WORD data_size, WORD timeout, BYTE data)
{
    BYTE tx_buf[12];
    BYTE *data_ptr;
    WORD buf_size;
    
    data_ptr = tx_buf;
    buf_size = 12;

    *data_ptr++ = MSG_BEGIN;
    *data_ptr++ = FILL_TAG;
    *data_ptr++ = (BYTE)(start_add / 256);
    *data_ptr++ = (BYTE)(start_add % 256);
    *data_ptr++ = (BYTE)(data_size / 256);
    *data_ptr++ = (BYTE)(data_size % 256);
    *data_ptr++ = (BYTE)(timeout / 256);
    *data_ptr++ = (BYTE)(timeout % 256);
    *data_ptr++ = 0x00;
    *data_ptr++ = data;
    *data_ptr++ = MSG_TERM;
    *data_ptr = MSG_TERM;

#if ANTARES
    com_buf_clr();              // Clear out Anyares COM receive buffer.
#endif    
    SendStream(tx_buf, buf_size);
}  
 
// ****************************************************************************
// Transmit the Block Read Command 
// Input:        Begin reading at this start address in the tag.
//               Number of bytes to be read from tag.
//               Antenna timeout time.
// Return:       None.
// Side effects: None.
void BLReadTx(WORD start_add, WORD data_size, WORD timeout)
{
    BYTE tx_buf[10];
    BYTE *data_ptr;
    WORD buf_size;

    data_ptr = tx_buf;
    buf_size = 10;

    *data_ptr++ = MSG_BEGIN;
    *data_ptr++ = BLK_READ;
    *data_ptr++ = (BYTE)(start_add / 256);
    *data_ptr++ = (BYTE)(start_add % 256);
    *data_ptr++ = (BYTE)(data_size / 256);
    *data_ptr++ = (BYTE)(data_size % 256);
    *data_ptr++ = (BYTE)(timeout / 256);
    *data_ptr++ = (BYTE)(timeout % 256);
    *data_ptr++ = MSG_TERM;
    *data_ptr = MSG_TERM;

#if ANTARES
    com_buf_clr();              // Clear out Anyares COM receive buffer.
#endif    
    SendStream(tx_buf, buf_size);
}   

// ****************************************************************************
// Transmit Enter Colntinual Block Read mode Command 
// Input:        Begin read tag address.
//               Number of bytes to read from tag.
// Return:       None.
// Side effects: None.
void ContReadTx(WORD start_add, WORD data_size)
{
    BYTE tx_buf[8];
    BYTE *data_ptr;
    WORD buf_size;

    data_ptr = tx_buf;
    buf_size = 8;

    *data_ptr++ = MSG_BEGIN;
    *data_ptr++ = CON_READ_MODE;
    *data_ptr++ = (BYTE)(start_add / 256);
    *data_ptr++ = (BYTE)(start_add % 256);
    *data_ptr++ = (BYTE)(data_size / 256);
    *data_ptr++ = (BYTE)(data_size % 256);
    *data_ptr++ = MSG_TERM;
    *data_ptr = MSG_TERM;

#if ANTARES
    com_buf_clr();              // Clear out Anyares COM receive buffer.
#endif    
    SendStream(tx_buf, buf_size);
}

// ****************************************************************************
// Transmit a proteced mode Block Write Command 
// Input:        Begin write at this start address in the tag.
//               Number of bytes to be written from tag.
//               Antenna timeout time.
//               Pointer to the bytes to be written.
// Return:       None.
// Side effects: None.
void pro_BLWriteTx(WORD start_add, WORD data_size, WORD timeout, BYTE *data_buf)
{   // Preform a normal block write.
    do_BLWrite(start_add, data_size, timeout, data_buf, PRO_BL_WRITE);
}

// ****************************************************************************
// Transmit a normal Block Write Command 
// Input:        Begin write at this start address in the tag.
//               Number of bytes to be written from tag.
//               Antenna timeout time.
//               Pointer to the bytes to be written.
// Return:       None.
// Side effects: None.
void BLWriteTx(WORD start_add, WORD data_size, WORD timeout, BYTE *data_buf)
{   // Preform a normal block write.
    do_BLWrite(start_add, data_size, timeout, data_buf, BLK_WRITE);
}

// ****************************************************************************
// Transmit get tag's Identification number Command 
// Input:        Antenna timeout time.
// Return:       None.
// Side effects: None.
void Get_Tag_IDTx(WORD timeout)
{
    BYTE tx_buf[6];
    BYTE *data_ptr;
    WORD buf_size;

    data_ptr = tx_buf;
    buf_size = 6;

    *data_ptr++ = MSG_BEGIN;
    *data_ptr++ = GET_TAG_ID;
    *data_ptr++ = (BYTE)(timeout / 256);
    *data_ptr++ = (BYTE)(timeout % 256);
    *data_ptr++ = MSG_TERM;
    *data_ptr = MSG_TERM;

#if ANTARES
    com_buf_clr();              // Clear out Anyares COM receive buffer.
#endif    
    SendStream(tx_buf, buf_size);
}

// ****************************************************************************
// Transmit Tag Search Command 
// Input:        Antenna timeout time.
// Return:       None.
// Side effects: None.
void SearchTx(WORD timeout)
{
    BYTE tx_buf[6];
    BYTE *data_ptr;
    WORD buf_size;

    data_ptr = tx_buf;
    buf_size = 6;

    *data_ptr++ = MSG_BEGIN;
    *data_ptr++ = SEARCH_TAG;
    *data_ptr++ = (BYTE)(timeout / 256);
    *data_ptr++ = (BYTE)(timeout % 256);
    *data_ptr++ = MSG_TERM;
    *data_ptr = MSG_TERM;

#if ANTARES
    com_buf_clr();              // Clear out Anyares COM receive buffer.
#endif    
    SendStream(tx_buf, buf_size);
}

// ****************************************************************************
// Transmit the Change baud rate
// Input:        New baud rate value. (See Operators's Manual for antenna.)
// Return:       None.
// Side effects: None.
void BaudrateTx(WORD baudrate) 
{
    BYTE tx_buf[6];
    
    tx_buf[MSG_BEG_INX] = MSG_BEGIN;
    tx_buf[MSG_FUN_INX] = BAUD_CHANG;
    tx_buf[2] = (BYTE)(baudrate / 256);
    tx_buf[3] = (BYTE)(baudrate % 256);
    tx_buf[4] = MSG_TERM;
    tx_buf[5] = MSG_TERM;

#if ANTARES
    com_buf_clr();              // Clear out Anyares COM receive buffer.
#endif    
    SendStream(tx_buf, 6);
}

// ****************************************************************************
// Recevive response form Non-Contiguous read.
// Input:        Pointer to where the data can be place.
//               Do not more than the data size number of bytes.
// Return:       Read status.
// Side effects: None.
BYTE NCReadRx(BYTE *data_buf, WORD data_size)
{
   BYTE *rx_buf;
   BYTE status;
   WORD ii;

    //allocate buffer and initialize them
   rx_buf = build_buf;
   ClearBuf(rx_buf, data_size*2 + 4);

#if TEST_MUX32
   if(NOT com_rec_buf(rx_buf, 2, REC_FIRST_BYTE)) // Throw out the first two bytes. 
      status = COMM_FAIL;   // They are MUX32 address and stuff.
   else
#endif   
   // first check the first 2 bytes to find out what is expected
   // if even the first 2 bytes are not gotten, serial communication fails
   // if get 0xff in 2nd bytes, receive Error message
   // else may get right message
   if (NOT com_rec_buf(rx_buf, 2, REC_FIRST_BYTE))
   {
       status = COMM_FAIL;
   } else if (rx_buf[MSG_FUN_INX] EQ 0xff) 
   {
       if (NOT com_rec_buf(rx_buf+2, 4, REC_BYTES_TIME))
       {
           status = COMM_FAIL;
       } else 
       {
           status = rx_buf[3];
       }
   } else 
   {
   if (com_rec_buf(rx_buf + 2, data_size*2 + 2, REC_MORE_BYTES) AND 
       (rx_buf[MSG_BEG_INX] EQ MSG_BEGIN) AND (rx_buf[MSG_FUN_INX] EQ NONCO_READ))
       {
           status = OP_OK;
           for(ii = 0; ii < data_size; ii++) 
               *data_buf++ = rx_buf[3 + ii*2];
       } else
           status = COMM_INV;
   }

   return(status);
}
           
// ****************************************************************************
// Receive response from Non-Contiguous write.
// Input:        None.
// Return:       Was the command echoed?
// Side effects: None.
BYTE NCWriteRx(void)
{
   //MSG_BEGIN, NONCO_WRITE, MSG_TERM, MSG_TERM if OK
   //MSG_BEGIN, 0xFF, 0x00, Error_Code, MSG_TERM, MSG_TERM if wrong
    return(RxCheck(NONCO_WRITE));
}

// ****************************************************************************
// Receive response from Non-Contiguous configuration.
// Input:        None.
// Return:       Was the command echoed?
// Side effects: None.
BYTE NCConfRx(void)
{
    return(RxCheck(CONF_NONCO));
}

// ****************************************************************************
// Receive response from fill command.
// Input:        None.
// Return:       Was the command echoed?
// Side effects: None.
BYTE FillRx(void)
{
#if ANTARES
    im_standby_wait(500); // Wait a half a second.
#else
    delay(500);           // Wait a half a second.
#endif
    return(RxCheck(FILL_TAG));
}

// ****************************************************************************
// Receive response from Block Read.
// Input:        where to place the data.
//               Receive no more data than this. Note: BYTE received in a WORD.
// Return:       Command status.
// Side effects: None.
BYTE BLReadRx(BYTE *data_buf, WORD data_size)
{
   return(do_BLReadRx(data_buf, data_size, BLK_READ));
}

// ****************************************************************************
// Receive response from Continual Block Read.
// Input:        where to place the data.
//               Receive no more data than this. Note: BYTE received in a WORD.
// Return:       Command status.
// Side effects: None.
BYTE ContBLReadRx(BYTE *data_buf, WORD data_size)
{
   return(do_BLReadRx(data_buf, data_size, CON_READ_MODE));
}

           
// ****************************************************************************
// Receive response from block write.
// Input:        None.
// Return:       Was the command echoed?
// Side effects: None.
BYTE BLWriteRx(void)
{
    return(RxCheck(BLK_WRITE));
}

// ****************************************************************************
// Receive response get tag's Identification number command.
// Input:        None.
// Return:       Was the command echoed? Tag present?
// Side effects: None.
BYTE Get_Tag_IDRx(BYTE *data_buf)
{
    return(do_BLReadRx(data_buf, ID_NUM_SIZE, GET_TAG_ID));
} 

// ****************************************************************************
// Receive response from search tag.
// Input:        None.
// Return:       Was the command echoed? Tag present?
// Side effects: None.
BYTE SearchRx(void)
{
    return(RxCheck(SEARCH_TAG));
} 

// ****************************************************************************
// Receive response from Protected Write Block command.
// Input:        None.
// Return:       Was the command echoed? Tag present?
// Side effects: None.
BYTE pro_BLWriteRx(void)
{
    return(RxCheck(PRO_BL_WRITE));
} 

// ****************************************************************************
// Receive response from Enter Continual Read mode command.
// Input:        None.
// Return:       Was the command echoed? Tag present?
// Side effects: None.
BYTE ContReadRx(void)
{
    return(RxCheck(CON_READ_MODE));
} 
