/////////////////////////////////////////////////////////////////////////////
// 10/14/01 - CFO.c: Custom File Open dialog.
// Copyright (C) 2001 by Creative Systems Software. All rights reserved.
// Written by Shawn Fessenden

// You are free to use, copy and distribute this source code as you see fit.
// Please retain original copyright notice in all modified versions.

// Windows defines.
#define STRICT							//Use strict types.
#define WIN32_LEAN_AND_MEAN	//Disinclude fluff.
#define _WIN32_WINNT 0x0500	//Compile for Windows 2000.

// Windows includes.
#include <windows.h>				//Base Windows.
#include <commdlg.h>				//Common dialogs.
#include <commctrl.h>				//Common controls (for common control messages & structures).

// CRT stuff.
#include <malloc.h>

// Export macro.
#define DllExport __declspec(dllexport)

/////////////////////////////////////////////////////////////////////////////
// GLOBALS: Buffers for user items.
#define BUFLEN (MAX_PATH + 1)	//Internal string buffers are 261 chars.
INT		giView = 3;							//0-based index of view (3 = Details view).
INT		giSort = 3;							//0-based index of sort (3 = Modified column).
BOOL	gbDecending = TRUE;			//If true, clicks column header twice.
BOOL	gbCenterWindow = TRUE;	//If true, centers dialog.
CHAR	gpTitle[BUFLEN];				//Dialog title.
CHAR	gpFile[BUFLEN];					//Selected path/file.
CHAR	gpFilter[BUFLEN];				//Filter(s).
CHAR	gpFiletitle[BUFLEN];		//Selected file.
CHAR	gpInitDir[BUFLEN];			//Initial directory.
CHAR	gpDefExt[4];						//Default extension.

// GLOBALS: Internal stuff.
HINSTANCE			ghInst = NULL;		//Instance handle of calling application.
OPENFILENAME	gOfn;							//Our open file name structure.

// Internal Function Prototypes.
// HookProc is the common dialog hook proceedure.
// MouseThread does all the automatic clicking.
// CheckCb checks return buffer lengths.
// Return returns strings to caller.
// Click clicks window hWnd at point ppt.
UINT_PTR CALLBACK	HookProc(HWND hDlg, UINT uiMsg, WPARAM wParam, LPARAM lParam);
DWORD WINAPI			MouseThread(PVOID pParam);
BOOL							CheckCb(PCHAR pItem, INT cb);
BOOL							Return(PCHAR pSrc, PCHAR pDst, INT cb);
VOID							Click(HWND hWnd, PPOINT ppt);

// Exported Function Prototypes.
DllExport BOOL	SelectFile(PCHAR pTitle, PCHAR pFilename, INT cbBufSize);
DllExport VOID	InitMembers(VOID);
DllExport BOOL	SetTitle(PCHAR pTitle);
DllExport BOOL	SetInitFile(PCHAR pFile);
DllExport INT		GetFilename(PCHAR pFilename, INT cbBufSize);
DllExport	INT		GetFiletitle(PCHAR pFiletitle, INT cbBufSize);
DllExport BOOL	SetFilter(PCHAR pFilter);
DllExport INT		GetFilter(PCHAR pFilter, INT cbBufSize);
DllExport VOID	SetFilterIndex(INT iFilterIndex);
DllExport BOOL	SetInitDir(PCHAR pInitDir);
DllExport INT		GetInitDir(PCHAR pInitDir, INT cbBufSize);
DllExport BOOL	SetDefExt(PCHAR pDefExt);
DllExport INT		GetDefExt(PCHAR pDefExt, INT cbBufSize);
DllExport VOID	SetFlags(UINT uFlags);
DllExport UINT	GetFlags(VOID);
DllExport	BOOL	SetView(INT iView);
DllExport	BOOL	SetSort(INT iSort);
DllExport VOID	SetCenter(INT iCenter);

/////////////////////////////////////////////////////////////////////////////
// ENTRY POINT.
BOOL WINAPI DllMain(HINSTANCE hInst, DWORD dwReason, LPVOID lpReserved)
{
	switch(dwReason) {
		case DLL_PROCESS_ATTACH: ghInst = hInst; InitMembers();
		case DLL_PROCESS_DETACH: ghInst = NULL;
	}
	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// Exported functions.
DllExport BOOL SelectFile(PCHAR pTitle, PCHAR pFilename, INT cbBufSize)
{
	BOOL	bRes = FALSE;

	// If a title was passed, use it.
	if(pTitle) lstrcpy(gpTitle, pTitle);

	// Display the dialog.
	bRes = GetOpenFileName(&gOfn);

	// Return file name to VEE.
	if(pFilename) Return(gpFile, pFilename, cbBufSize);
	return bRes;
}

DllExport VOID InitMembers()
{
	// Initialize OFN.
	ZeroMemory(&gOfn, sizeof(gOfn));
	gOfn.lStructSize = sizeof(gOfn);
	gOfn.hInstance = ghInst;
	gOfn.lpstrFile = gpFile;
	gOfn.lpstrFileTitle = gpFiletitle;
	gOfn.lpstrTitle = gpTitle;
	gOfn.nMaxFile = MAX_PATH;
	gOfn.Flags = OFN_ENABLEHOOK | OFN_EXPLORER;
	gOfn.lpfnHook = HookProc;

	// Clear buffers.
	ZeroMemory(gpTitle, BUFLEN);
	ZeroMemory(gpFile, BUFLEN);
	ZeroMemory(gpFilter, BUFLEN);
	ZeroMemory(gpFiletitle, BUFLEN);
	ZeroMemory(gpInitDir, BUFLEN);
	ZeroMemory(gpDefExt, 4);
}

DllExport BOOL SetTitle(PCHAR pTitle)
{
	if(CheckCb(pTitle, BUFLEN)) {
		lstrcpy(gpTitle, pTitle);
		return TRUE;
	}
	return FALSE;
}

DllExport BOOL	SetInitFile(PCHAR pFile)
{
	if(CheckCb(pFile, BUFLEN)) {
		lstrcpy(gpFile, pFile);
		return TRUE;
	}
	return FALSE;
}

DllExport INT		GetFilename(PCHAR pFilename, INT cbBufSize)
{
	if(Return(gpFile, pFilename, cbBufSize)) return lstrlen(gpFile);
	return 0;
}

DllExport	INT		GetFiletitle(PCHAR pFiletitle, INT cbBufSize)
{
	if(Return(gpFiletitle, pFiletitle, cbBufSize)) return lstrlen(gpFiletitle);
	return 0;
}

DllExport BOOL	SetFilter(PCHAR pFilter)
{
	int		iLen = 0;
	CHAR	pDelim[] = "|";

	PCHAR	p = NULL;
	PCHAR	p1 = NULL;
	PCHAR	pBuf = NULL;

	// Can't exceed MAX_PATH + 2.
	if(CheckCb(pFilter, BUFLEN + 1)) {
		
		// Get a local buffer & copy input string.
		if((pBuf = malloc(lstrlen(pFilter) + 2)) != NULL) {
			lstrcpy(pBuf, pFilter);
			
			// Find vertical bars and replace them with NULLs.
			p = strtok(pBuf, pDelim);
			while(p) {p1 = p; strtok(NULL, pDelim);}

			// Get to the end of the last filter string and double zero terminate it.
			while(*p1++); p1++; *p1 = 0;

			// Now, copy the completed filter to gpFilter & point lpstrFilter to it.
			iLen = p1 - p;
			memcpy(gpFilter, p, iLen);
			gOfn.lpstrFilter = gpFilter;
			
			// Set the default index to 1, clean up & return true.
			gOfn.nFilterIndex = 1;
			free(p);
			return TRUE;
		}
	}

	// Problem: filter too long or malloc failed.
	return FALSE;
}

DllExport INT		GetFilter(PCHAR pFilter, INT cbBufSize)
{
	int		i, iLen = 0;
	CHAR	pDelim[] = "|";

	PCHAR	p = NULL;
	PCHAR pBuf = NULL;

	// First, we have to reverse the "damage" done by SetFilter.
	if((p = gpFilter) != NULL) {

		// Find the length of the double zero terminated string.
		while(TRUE) {
			if(*p == 0) {if(*(p + 1) == 0) break; p++;}
			iLen++;
		}
		iLen++;

		// Copy it to a local buffer, replace NULLs with a vertical bar & return it.
		if((pBuf = malloc(iLen)) != NULL) {
			memcpy(pBuf, gpFilter, iLen);
			for(i = 0; i < iLen; i++) {if(pBuf[i] = 0) pBuf[i] = '|';}
			if(Return(pBuf, pFilter, cbBufSize)) return iLen - 1;
			free(pBuf);
		}
	}
	
	// Problem: no filter, malloc failed or user buffer too small.
	return 0;
}

DllExport VOID	SetFilterIndex(INT iFilterIndex)
{gOfn.nFilterIndex = iFilterIndex;}

DllExport BOOL	SetInitDir(PCHAR pInitDir)
{
	if(CheckCb(pInitDir, BUFLEN)) {
		lstrcpy(gpInitDir, pInitDir);
		gOfn.lpstrInitialDir = gpInitDir;
		return TRUE;
	}
	return FALSE;
}

DllExport INT		GetInitDir(PCHAR pInitDir, INT cbBufSize)
{
	if(Return(gpInitDir, pInitDir, cbBufSize)) return lstrlen(gpInitDir);
	return 0;
}

DllExport BOOL	SetDefExt(PCHAR pDefExt)
{
	if(CheckCb(pDefExt, 4)) {
		lstrcpy(gpDefExt, pDefExt);
		gOfn.lpstrDefExt = gpDefExt;
		return TRUE;
	}
	return FALSE;
}

DllExport INT		GetDefExt(PCHAR pDefExt, INT cbBufSize)
{
	if(Return(gpDefExt, pDefExt, cbBufSize)) return lstrlen(gpDefExt);
	return 0;
}

DllExport VOID	SetFlags(UINT uFlags)
{gOfn.Flags = uFlags;}

DllExport UINT	GetFlags(VOID)
{return gOfn.Flags;}

DllExport BOOL	SetView(INT iView)
{
	if(iView > -1 && iView < 5) {
		giView = iView;
		return TRUE;
	}
	return FALSE;
}

DllExport BOOL	SetSort(INT iSort)
{
	if(iSort > -1 && iSort < 4) {
		giSort = iSort;
		return TRUE;
	}
	return FALSE;
}

DllExport VOID	SetCenter(INT iCenter)
{
	if(iCenter == 0) gbCenterWindow = FALSE;
	else gbCenterWindow = TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// Helper functions.
BOOL CheckCb(PCHAR pItem, INT cb)
{
	if(pItem) return lstrlen(pItem) < cb;
	return FALSE;
}

BOOL Return(PCHAR pSrc, PCHAR pDst, INT cb)
{
	if(!CheckCb(pSrc, cb)) return FALSE;
	lstrcpy(pDst, pSrc);
	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// Callbacks.
UINT_PTR CALLBACK HookProc(HWND hDlg, UINT uiMsg, WPARAM wParam, LPARAM lParam)
{
	int					iWidth, iHeight;
	RECT				rc;
	HWND				hWnd = NULL;
	DWORD				dwThreadId = 0;
	HANDLE			hMouseThread = NULL;
	LPOFNOTIFY	pNotify = (LPOFNOTIFY)lParam;

	// Wait for CDN_INITDONE message.
	if(uiMsg == WM_NOTIFY) {
		if(pNotify->hdr.code == CDN_INITDONE) {

			// If centering was requested, center dialog.
			if(gbCenterWindow) {
				hWnd = GetParent(hDlg);
				GetWindowRect(hWnd, &rc);
				iWidth = rc.right - rc.left; iHeight = rc.bottom - rc.top;
				rc.left = (GetSystemMetrics(SM_CXSCREEN) - iWidth) / 2;
				rc.top = (GetSystemMetrics(SM_CYSCREEN) - iHeight) / 2;
				MoveWindow(hWnd, rc.left, rc.top, iWidth, iHeight, FALSE);
			}

			// If we're going to click anything, start the mouse thread.
			if(!(giView == 2 && giSort == 0)) {
				hMouseThread = CreateThread(NULL, 0, MouseThread, NULL, 0, &dwThreadId);
				CloseHandle(hMouseThread);
			}
		}
	}
	return 0;
}

/////////////////////////////////////////////////////////////////////////////
// Aux threads.
DWORD WINAPI MouseThread(LPVOID pParam)
{
	HWND	hDlg = NULL;			//Dialog's handle.
	HWND	hList = NULL;			//List view's handle.
	HWND	hShell = NULL;		//Parent of list view.
	HWND	hHeader = NULL;		//List view's header control's handle.
	HWND	hToolbar = NULL;	//Toolbar's handle.
	RECT	rcViewButton;			//Toolbar's view button.
	RECT	rcHeaderColumn;		//Header's column to click.
	POINT	ptMouse;					//Mouse click coordinates.

	// Find the toolbar.
	hDlg = FindWindow(NULL, gpTitle);
	hToolbar = FindWindowEx(hDlg, NULL, "ToolbarWindow32", NULL);

	// Get view menu button stuff & click button.
	SendMessage(hToolbar, TB_GETITEMRECT, (WPARAM)3, (LPARAM)&rcViewButton);
	ptMouse.x = rcViewButton.left + 5; ptMouse.y = rcViewButton.top + 5;
	Click(hToolbar, &ptMouse);

	// Figure out where to click for selected view style & click.
	ptMouse.x = rcViewButton.left + 20;
	ptMouse.y = rcViewButton.bottom + 10 + (giView * GetSystemMetrics(SM_CYMENU));
	ClientToScreen(hToolbar, &ptMouse);
	Click(hToolbar, &ptMouse);

	// Now, if we're clicking a column...
	if(giSort != 0 || (giSort = 0 && gbDecending)) {

		// Wait for all this to happen.
		while(hHeader == NULL) {
			hShell = FindWindowEx(hDlg, NULL, "SHELLDLL_DefView", NULL);
			hList = FindWindowEx(hShell, NULL, "SysListView32", NULL);
			hHeader = ListView_GetHeader(hList);
		}

		// Now, figure out where to click for selected sort.
		Header_GetItemRect(hHeader, giSort, &rcHeaderColumn);
		ptMouse.x = rcHeaderColumn.left + ((rcHeaderColumn.right - rcHeaderColumn.left) / 2);
		ptMouse.y = rcHeaderColumn.top  + ((rcHeaderColumn.bottom - rcHeaderColumn.top) / 2);

		// Click if not sorting on Name column.
		if(giSort != 0) Click(hHeader, &ptMouse);

		// If decending order is requested, click again.
		if(gbDecending) Click(hHeader, &ptMouse);
	}

	ExitThread(0);
	return 0;
}

void Click(HWND hWnd, PPOINT ppt)
{
	LPARAM	lParam = MAKELPARAM(ppt->x, ppt->y);
	PostMessage(hWnd, WM_LBUTTONDOWN, MK_LBUTTON, lParam);
	PostMessage(hWnd, WM_LBUTTONUP,            0, lParam);
}
