﻿using System;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;

namespace SorterLibrary
{
    // Code copied from http://support.microsoft.com/kb/319401

    public enum SortTypes
    {
        SortTextCaseInsensitive = 0,
        SortText = 3,
        SortNumeric = 1,
        SortDateTime = 2,
        SortCustom = 4
    }

    public enum CompareResults
    {
        GreaterThan = 1,
        LessThan = -1,
        EqualsTo = 0
    }

    public delegate void ItemSortEventHandler(ListViewColumnSorter sender, ItemSortEventArgs e);

    /// <summary>
    /// This class is an implementation of the 'IComparer' interface.
    /// </summary>
    [Description("Helper Column Sorting Class for .Net ListView objects"), Category("Default")]
    public class ListViewColumnSorter : IComparer
    {
        /// <summary>
        /// Specifies the column to be sorted
        /// </summary>
        private int ColumnToSort;
        /// <summary>
        /// Specifies the order in which to sort (i.e. 'Ascending').
        /// </summary>
        private SortOrder OrderOfSort;
        /// <summary>
        /// Case insensitive comparer object
        /// </summary>
        //private CaseInsensitiveComparer ObjectCompare;

        private SortTypes TypeOfSort;
        /// <summary>
        /// The Object used for the Tag
        /// </summary>
        private Object TagObject;
        /// <summary>
        /// Result that is returned when using SortCustom and the SortItems Event
        /// </summary>
        private CompareResults CompareEventResult;

        public event ItemSortEventHandler OnItemSort;

        /// <summary>
        /// Class constructor.  Initializes various elements
        /// </summary>

        public ListViewColumnSorter()
        {
            // Initialize the column to '0'
            ColumnToSort = 0;

            // Initialize the sort order to 'Ascending'
            OrderOfSort = SortOrder.None;

            // Initialize objects
            //ObjectCompare = new CaseInsensitiveComparer();

            SortType = SortTypes.SortText;
            CompareEventResult = CompareResults.EqualsTo;
        }

        /// <summary>
        /// This method is inherited from the IComparer interface.  It compares the two objects passed using a case insensitive comparison.
        /// </summary>
        /// <param name="x">First object to be compared</param>
        /// <param name="y">Second object to be compared</param>
        /// <returns>The result of the comparison. "0" if equal, negative if 'x' is less than 'y' and positive if 'x' is greater than 'y'</returns>
        public int Compare(object x, object y)
        {
            int compareResult;
            double Value1, Value2;
            DateTime Dval1, Dval2;

            ListViewItem listviewX, listviewY;
            
            // Cast the objects to be compared to ListViewItem objects
            listviewX = (ListViewItem)x;
            listviewY = (ListViewItem)y;
            compareResult = 0;

            // Compare the two items
            switch ((int) TypeOfSort)
            {
                case (int)SortTypes.SortTextCaseInsensitive:
                    compareResult = string.Compare(listviewX.SubItems[ColumnToSort].Text, listviewY.SubItems[ColumnToSort].Text, true);
                    //compareResult = ObjectCompare.Compare(listviewX.SubItems[ColumnToSort].Text, listviewY.SubItems[ColumnToSort].Text);
                    break;
                    
                case (int)SortTypes.SortText:
                    
                    compareResult = string.CompareOrdinal(listviewX.SubItems[ColumnToSort].Text, listviewY.SubItems[ColumnToSort].Text);
                   // compareResult = Comparer.DefaultInvariant.Compare(listviewX.SubItems[ColumnToSort].Text, listviewY.SubItems[ColumnToSort].Text);
                    break;

                case (int)SortTypes.SortNumeric:
                
                    Value1 = double.Parse(listviewX.SubItems[ColumnToSort].Text);
                    Value2 = double.Parse(listviewY.SubItems[ColumnToSort].Text);

                    if (Value1 > Value2) compareResult = 1;
                    else if (Value1 == Value2) compareResult = 0;
                    else compareResult = -1;
                    break;

                case (int)SortTypes.SortDateTime:

                    Dval1 = DateTime.Parse(listviewX.SubItems[ColumnToSort].Text);
                    Dval2 = DateTime.Parse(listviewY.SubItems[ColumnToSort].Text);

                    if (Dval1 > Dval2) compareResult = 1;
                    else if (Dval1 == Dval2) compareResult = 0;
                    else compareResult = -1;
                    break;

                case (int) SortTypes.SortCustom:

                    DoItemSort(new ItemSortEventArgs(ColumnToSort, listviewX, listviewY));
                    compareResult = (int) CompareEventResult;
                    break;
            }

            // Calculate correct return value based on object comparison
            if (OrderOfSort == SortOrder.Ascending)
            {
                // Ascending sort is selected, return normal result of compare operation
                return compareResult;
            }
            else if (OrderOfSort == SortOrder.Descending)
            {
                // Descending sort is selected, return negative result of compare operation
                return (-compareResult);
            }
            else
            {
                // Return '0' to indicate they are equal
                return 0;
            }
        }

        protected virtual void DoItemSort(ItemSortEventArgs e)
        {
            if (OnItemSort != null) OnItemSort(this, e);   
        }

        /// <summary>
        /// Gets or sets the number of the column to which to apply the sorting operation (Defaults to '0').
        /// </summary>
        [Description("Specifies which column to sort upon.  Defaults to 0")]
        public int SortColumn
        {
            set
            {
                ColumnToSort = value;
            }
            get
            {
                return ColumnToSort;
            }
        }

        /// <summary>
        /// Gets or sets the order of sorting to apply (for example, 'Ascending' or 'Descending').
        /// </summary>
        [Description ("Specifies whether to sort Ascending (1) or Descending (2).  Defaults to Descending"), Category("Default")]
        public SortOrder Order
        {
            set
            {
                OrderOfSort = (SortOrder) Enum.Parse(typeof(SortOrder), value.ToString());
            }
            get
            {
                return OrderOfSort;
            }
        }
        /// <summary>
        /// 
        /// </summary>
        [Description("Specifies whether to column entries as numbers (nonzero) or as text (0).  Defaults to Text")]
        public SortTypes SortType
        {
            set
            {
                TypeOfSort = value;
            }
            get
            {
                return TypeOfSort;
            }
        }

        /// <summary>
        /// Gets or sets the Tag Property
        /// </summary>
        public Object Tag
        {
            get
            {
                return TagObject;
            }
            set
            {
                TagObject = value;
            }
        }

        /// <summary>
        /// Gets or sets the Result Returned when the SortItems Event is handled
        /// </summary>
        public CompareResults CompareResult
        {
            get
            {
                return CompareEventResult;
            }
            set
            {
                CompareEventResult = value;
            }
        }
    }

    public class ItemSortEventArgs : EventArgs
    {
        private int Column;
        private ListViewItem ListA;
        private ListViewItem ListB;
    
        public ItemSortEventArgs(int ColumnToSort, ListViewItem ListItemA, ListViewItem ListItemB)
        {
            Column = ColumnToSort;
            ListA = ListItemA;
            ListB = ListItemB;
        }
        /// <summary>
        /// </summary>
        /// <remarks></remarks>
        /// <value></value>
        public int SortColumn
        {
            get
            {
                return Column;
            }
        }

        public ListViewItem ListViewItemA
        {
            get
            {
                return ListA;
            }
        }

        public ListViewItem ListViewItemB
        {
            get
            {
                return ListB;
            }
        }
    }

}
