// PV.CPP: Display / Modify privileges.
// Copyright (C) 1999 by Oswego Software, Inc. All rights reserved.
// Written by Shawn Fessenden.

#define WIN32_LEAN_AND_MEAN
#include <windows.h>

#include <tchar.h>
#include <stdio.h>
#include <iostream.h>

// GLOBALS.
static TCHAR pSignOnMsg[] = {
	_T("pv.exe Version 0.8b: Display / Modify privilege information.\n")
	_T("Copyright (C) 1999 by Oswego Software, Inc. All rights reserved.\n")
	_T("Check us out at www.oswegosw.com\n\n")
};

static TCHAR pHelpMsg[] = {
	_T("USAGE:\n")
	_T("pv [e <PrivilegeName> | d <PrivilegeName> | <PrivilegeName> | h | ?]\n\n")
	_T("  e <PrivilegeName>: Enable specified privilege.\n")
	_T("  d <PrivilegeName>: Disable specified privilege.\n")
	_T("  <PrivilegeName>  : Display status of specified privilege.\n\n")
	_T("Switch delimiters (- or /) are optional.\n")
	_T("If no command line is passed, pv displays the status of all held privileges.\n")
	_T("Not all logins will be able to modify privileges.\n")
};

static DWORD pdwAccess[] = {
	TOKEN_ALL_ACCESS,												//Try this first.
	TOKEN_QUERY | TOKEN_ADJUST_PRIVILEGES,	//Need at least this level of access to adjust.
	TOKEN_QUERY															//Display only.
};

static PTCHAR pPrivMsgs[] = {
	_T("This account has basically super user control over it's privileges.\n\n"),
	_T("This account will be able to modify what privileges it holds.\n\n"),
	_T("This account will not be able to modify it's privileges.\n\n")
};

static PBYTE		gptp					= NULL;
static HANDLE		ghToken				= NULL;

// Command line vars.
static bool			gbHelp				= false;
static bool			gbEnablePriv	= false;
static bool			gbDisablePriv	= false;
static PTSTR		gpszPrivName	= NULL;

// FUNCTIONS.
void SignOn(void)
{cout << pSignOnMsg;}

void PrintHelp(void)
{cout << pHelpMsg;}

bool ProcessCmdLine(int argc, char *argv[], char *envp[])
{
	int		iNameLen;
	char	c, *p;

	// Interpret command line switches (if any).
	if(argc < 2) return true;

	// We don't actually need any switch char ('-' or '/') because the CLP has already
	// parsed out nice tidy strings for us. However, this is the way things are done so
	// we either look for '-' or '/' or assume the first character is a command.
	for(int i = 1; i < argc; i++) {
		p = argv[i];
		if(*p == '-' || *p == '/') p++;

		// Get command.
		c = tolower(*p);

		// If the next char is a colon or a space, skip it.
		p++; if(*p == ':' || *p == ' ') p++;

		switch(c) {

			// Disable privilege.
			case 'd':	
				iNameLen = lstrlen(p);
				gpszPrivName = new TCHAR[iNameLen + 1];
				strcpy(gpszPrivName, p);
				gbDisablePriv = true;
			break;
			
			// Enable privilege.
			case 'e':
				iNameLen = lstrlen(p);
				gpszPrivName = new TCHAR[iNameLen + 1];
				strcpy(gpszPrivName, p);
				gbEnablePriv = true;
			break;

			// Help.
			case 'h':	gbHelp = true;		break;
			case '?':	gbHelp = true;		break;

			// Display single privilege status.
			default:
				p--;
				iNameLen = lstrlen(p);
				gpszPrivName = new TCHAR[iNameLen + 1];
				strcpy(gpszPrivName, p);
			break;
		}
	}

	// No error. Done.
	return true;
}

bool GetTokenPrivileges(DWORD dwAccess)
{
	DWORD		dwBytesRequired = 0;

	// If mem is allocated, get rid of it.
	if(gptp) {
		delete [] gptp;
		gptp = NULL;
	}

	// Get process token.
	if(!OpenProcessToken(GetCurrentProcess(), dwAccess, &ghToken)) return false;

	// Get token information.
	GetTokenInformation(ghToken, TokenPrivileges, NULL, 0, &dwBytesRequired);
	gptp = new BYTE[dwBytesRequired];
	if(!GetTokenInformation(ghToken, TokenPrivileges, gptp, dwBytesRequired, &dwBytesRequired)) return false;

	return true;
}

void DisplayTokenPrivileges()
{
	bool		bOnlyOne = false;
	bool		bPrintIt = false;
	TCHAR		pName[81];
	TCHAR		pMsgLine[81];
	DWORD		dwNameSize, i;

	// If there's a privilege name in gpszPrivName, then display only that privilege.
	if(gpszPrivName) bOnlyOne = true;

	// Walk the LUID_AND_ATTRIBUTES structures of the TOKEN_PRIVILEGES returned & log them.
	for(i = 0; i < ((PTOKEN_PRIVILEGES)gptp)->PrivilegeCount; i++) {
		bPrintIt = false;
		dwNameSize = 80;
		if(!LookupPrivilegeName(_T(""), &((PTOKEN_PRIVILEGES)gptp)->Privileges[i].Luid, pName, &dwNameSize)) {
			cout << _T("LookupPrivilegeName failed with ") << GetLastError() << endl;
		} else {
			if(!bOnlyOne) bPrintIt = true;
			else if(strcmp(gpszPrivName, pName) == 0) bPrintIt = true;
			if(bPrintIt) {
				sprintf(pMsgLine, _T("%- 32s: %s"), pName,
					(((PTOKEN_PRIVILEGES)gptp)->Privileges[i].Attributes ? _T("Enabled") : _T("Disabled")));
				cout << pMsgLine << endl;
			}
		}
	}
}


int main(int argc, char *argv[], char *envp[])
{
	int	i;

	// Print sign on message.
	SignOn();

	// Process command line options.
	if(!ProcessCmdLine(argc, argv, envp)) return 1;

	// Print help if requested.
	if(gbHelp) {
		PrintHelp();
		return 0;
	}

	// Fill token privileges. May take a few tries.
	for(i = 0; i < sizeof(pdwAccess) / sizeof(DWORD); i++) {
		if(GetTokenPrivileges(pdwAccess[i])) break;
	}

	// Report access level.
	cout << pPrivMsgs[i];

	// Abort if we didn't get token privileges.
	if(!gptp) return 2;

	// Show 'em.
	DisplayTokenPrivileges();

	// Clean up & exit.
	if(gpszPrivName)	delete [] gpszPrivName;
	if(gptp)					delete [] gptp;
	if(ghToken)				CloseHandle(ghToken);

	return 0;
}
