#!/usr/bin/perl
# -------------------------------------------------------------------------
#
#    VeeColorizer - adds color info to Agilent VEE (tm) programs
#
#    Copyright (C) 2001 Rolf Eichenseher <roffe@gmx.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
#
# -------------------------------------------------------------------------


use Getopt::Std;

$PROGNAME   = "VEE Colorizer";
$VERSION    = "0.92";
$AUTHOR     = "Rolf Eichenseher <roffe\@gmx.net>";
$DATE       = "15-May-2001";
$COLORTABLE = "colors.ini";


# -------------------------------------------------------------------------
# parseCmdLine()
# 
# Check the command line for switches and switch combinations,
# set the appropriate variables ($opt_*) or $opt_n if no switch is 
# specified (=(n)ormal operation). Input file name(s) are 'globbed' and
# stored in @ARGV.
# The path where this program resides is stored in $path 
# (that we know where to look for colortable.txt)
# -------------------------------------------------------------------------

sub parseCmdLine
{
   usage() if ( $#ARGV == -1 );
   getopts( "c:fqu" );
   $opt_n = ( !$opt_u && !$opt_f );  # set opt_n when in "normal" operation
   die( "Error: illegal switch combination -u with -f" ) if ( $opt_u && $opt_f );
   reverse( $0 ) =~ /[^\/:]*(.*)/;   # determine installation path of this program
   $path = reverse( $1 );         
}   


# -------------------------------------------------------------------------
# usage()
#
# Print program help and exit
# -------------------------------------------------------------------------

sub usage
{
   print "\n$PROGNAME $VERSION\n".
         "$AUTHOR $DATE\n\n".
         "Adds (or removes) color info to Agilent (tm) VEE objects.\n".
         "Usage:\n".
         "  veecolorizer.pl [-<options>] <inputfile(s) [...]> \n\n".
         "<inputfile(s) [...]> can be a list of one or more files or a set\n".
         "  of files specified by wildcards (*/?).\n\n".        
         "<options> can be one ore more of the following:\n".
         "  -c <file> : read color table from <file> (default: $COLORTABLE)\n".
         "  -f        : force (overwrite existing colors)\n".
         "  -q        : be quiet (don't print any information)\n".
         "  -u        : uncolorize (remove color info from all objects)\n";
   exit;
}
       

# -------------------------------------------------------------------------
# getVersion
#
# Scans <INFILE> for a line of the form "(saveFormat "n.n")" and returns
# the save format version number "n.n" or "(unknown)" if the version number
# can not be determined.
# -------------------------------------------------------------------------

sub getVersion
{
   my ($line);

   while ( $line = <INFILE> )
   {
      return $1 if ( $line =~ /\(saveFormat \"([0-9.]*)\"\)/ );
   }
   return "(unknown)";
}
              

# -------------------------------------------------------------------------
# loadColorTable( filename )
#
# Loads the color table from <filename> into the $colors hash.
# $colors is of the following form:
#   - Keys are object classes
#   - Values are associated colors
# -------------------------------------------------------------------------

sub loadColorTable
{
   open( COLORFILE, $_[0] ) or die ( "Can not open color table $_[0]: $!" );
   while ( <COLORFILE> )
   {
      next if /^#/;                             # ignore comment lines
      next unless /(\S*)                        # find object type,
                   \s*                          #   whitespaces
                   ("[a-zA-Z ]*\")              #   then background color (quoted)
                   \s*                          #   whitespaces
                   ("[a-zA-Z ]*\"|.*)/x;        #   foreground color (quoted, optional)

      $bgColors{$1} = $2;                       # add to hash: key=object type,
      $fgColors{$1} = $3;                       #              value=color
   }
}
      


# -------------------------------------------------------------------------
# getScopingChange( string )
#
# Determines if the scoping level has changed by counting opening and
# closing brackets in "string". Brackets within literal constants
# are ignored.
#
# Parameters: 
#   string: Line where to search for opening and closing brackets 
# Returns:
#   int:    Number of changes in the scoping level (number of leading 
#           opening brackets minus number of trailing closing brackets)
#           result == 0: scoping level unchanged
#           result > 0:  scoping level increased "result" times
#           result < 0:  scoping level decreased "result" times
# -------------------------------------------------------------------------

sub getScopingChange
{
   my ($string, $res, $c);

   $string = $_[0];
   while ( $string =~ /(.)/ )         # examine character-wise
   {
      $string = $';
      $c = $1;                        
      if (( $c =~ /\"/ ) && !$wasBackslash ) # string literal start/end?
      {
         $ignore = (!$ignore);
         next;
      }
      $wasBackslash = ( $c =~ /\\/ ) && ( !$wasBackslash ); 
      next if $ignore;
      $res++ if ( $c =~ /\(/ );
      $res-- if ( $c =~ /\)/ );
   }
   return $res;
}


# -------------------------------------------------------------------------
# colorizeContext( initialContext )
#
# Adds/removes color info for all objects in the current context.
# VEE Object identifiers are unique only within a particular context (e.g.
# within a UserObject). 
# Note that contexts can be recursively folded, so this function calls 
# itself if a subcontext is found.
#
# Parameters:
#    initialContext: number to initially set the scoping level.
#                    Must be 0 if the function is called from itself
#                    and 1 if the function is called from outside.
#                    (The reason for this is that VEE's Main() function
#                    is not inside a bracket pair, so we have to emulate
#                    "scoping level +1". If we omit this, "devCarrierFor"
#                    entries for objects inside Main() are not written.)
# Returns:
#   Number of scoping level changes (= number of times stepped out from 
#   context recursion, always negative)
# -------------------------------------------------------------------------

sub colorizeContext
{

   # Variables _must_ be local (recursively called->don't forget old values!)
   my ($dummy, $number, $type, %devices, $context, $fg, $bg);

   
   # -------------------------------------------------------
   # int changeObjectColor( backgroundColor [,foregroundColor] )
   #
   # Changes the color of the current object according to 
   # the given options.
   # This functions requires the <FILE> reading position to 
   # be set directly after the "(devCarrierFor nnn" line.
   #
   # Parameters:
   #   backgroundColor: (optional): background color
   #   foregroundColor: (optional): foreground color
   # If backgroundColor and foregroundColor are both
   # omitted and we are in force (-f) mode, any existing
   # color information will be removed.
   #
   # Returns:
   #   ($context, $modified)
   #   $context: int: scoping level change relative 
   # -------------------------------------------------------

   sub changeObjectColor
   {
      my ($context, $modified, $hasColor);    # local parenthesis counter = current scoping level 
  
      while ( <INFILE> )
      {         
         $context += getScopingChange( $_ );
         if ( /\(title.g.*/ )                 # is there already color info?
         {
            $hasColor = 1;
            if ( $opt_u || $opt_f )           # if in uncolorize or force mode...
            {
               $modified = 1;                 
               next;                          # don't write old color info. 
            }
         }

         if ( $opt_f || ( $opt_n && !$hasColor ))    
         {
            if (( $_[0] ) && ( /\(pinCenter [\-0-9]+ [\-0-9]+/ ))   # last line of "devCarrierFor"...?
            {
               $modified = 1;
               print "\(titleBg $_[0])\n";                # ...then insert color info
               print "\(titleFg $_[1])\n" if ( $_[1] );   # do we have foreground color info?
            }
         }
         print;                                           # write original line
      } 
      continue 
      {   
         return ($context, $modified) if ( $context < 0 );     # end of "devCarrierFor" ? 
      }
   } # end sub

  
 
   $context = $_[0];

   while ( <INFILE> )     # Read the VEE file and...
   {
      $context += getScopingChange( $_ );
      print;                                      # write original line
      return $context if ( $context < 0 );
     
      if ( /\(device (\d+)\s*(\S*)\b/ )           # object definition ? 
      {              
         $devices{$1}{type} = $2;                 # store object id & type
         if (( $2 eq "CONTEXT" ) || ( $2 eq "ROOTCONTEXT" ))  # is device a subcontext ?
         {
             $context += colorizeContext( 0 );       
             return $context if ( $context < 0 );
         }
      }
      elsif ( /\(devCarrierFor (\d+)\b/ )      # object properties ? 
      {

         $bg = $bgColors{$devices{$1}{type}};  # look up background color
         $fg = $fgColors{$devices{$1}{type}};  # look up foreground color
         ($contextChange, $modified) = changeObjectColor( $bg, $fg );  # set current object to color
         $context += $contextChange;
         return $context if ( $context < 0 );

         if ( $modified )
         {
            print STDOUT "." unless $opt_q;    # object was changed
            $count++;
         }
      }

   }  # end while
}  # end sub

  

# -------------------------------------------------------------------------
# main( [options] veefile(s) [...] )
# -------------------------------------------------------------------------


parseCmdLine();
loadColorTable( $opt_c || $path.$COLORTABLE );
die( "Error: no input file(s) specified" ) if ( $#ARGV == -1 );

# colorize each file
foreach ( @ARGV )                          
{
   $count = 0;
   $infile = $_;
   print "Processing $infile: " unless $opt_q;

   open(INFILE, $_) or die( "Error: could not open $infile: $!" );

   if (( $version = getVersion()) < "2.3" )
   {
      print "Error: save format $version not supported.\n";
      close INFILE;
      next;
   }
   # be sure to read again from the start
   seek( INFILE, 0, 0 );   

   # read original file and write output to a temporary file
   s/(.*)(\.vee)/$1/i;                     
   open(OUTFILE, ">$1\.tmp") or die( "Error: could not open $1\.tmp: $!" );
   select OUTFILE;
   
   colorizeContext( 1 );

   # postproc: cleanup
   close INFILE;
   close OUTFILE;
   select STDOUT;
   print "$count object(s) changed.\n" unless $opt_q;

   # rename original file to ".colorizer.bak" 
   rename $infile, "$1\.colorizer.bak" or die( "Could not rename $infile to $1\.colorizer.bak: $!" );

   # rename temporary file to original file
   rename "$1\.tmp", $infile or die( "Could not rename $1\.tmp to $infile: $!" );
}