REM Microsoft Q-Basic source code.
REM GENDOC.BAS - Generates documentation for VEE user functions.
REM Copyright (C) 1999 by Creative Systems Software.
REM Written by Shawn Fessenden.
REM Exclusive use rights leased to Oswego Software, Inc.
REM
REM Copyright (C) 2004 Black Cat Software.
REM Released to public domain 05/27/2004.

DECLARE SUB NormalizeInputFile ()
DECLARE FUNCTION CountRightParens% (strLine AS STRING)

DECLARE SUB AnalyzeFunctionData ()
DECLARE FUNCTION GetFunctionInfo% ()
DECLARE FUNCTION AllocFunc% (strNameData AS STRING)
DECLARE FUNCTION GetFunctionParam% (iDirection AS INTEGER)
DECLARE FUNCTION AllocParam% (strNameData AS STRING)

DECLARE SUB WriteOutputFile ()
DECLARE FUNCTION GetParamType$ (iType AS INTEGER)
DECLARE FUNCTION GetParamShape$ (iShape AS INTEGER)

' This program generates documentation for VEE user functions.
' Ya know, with slight tweaks this could also generate header files.

' Standard definitions of true & false.
CONST FALSE = 0
CONST TRUE = NOT FALSE

' Limits
CONST MAX.FUNCS = 500               '
CONST MAX.FUNCNAME = 50             '
CONST MAX.PARAMSPERFUNC = 5         '
CONST MAX.PARAMNAME = 20            '

' VEE Variable types: VT_
CONST VT.UNKNOWN = 0                '
CONST VT.INT32 = 1                  '
CONST VT.REAL = 2
CONST VT.PCOMPLEX = 3               '
CONST VT.COMPLEX = 4                '
CONST VT.WAVEFORM = 5               '
CONST VT.SPECTRUM = 6               '
CONST VT.COORD = 7                  '
CONST VT.TEXT = 8                   '
CONST VT.ENUM = 9                   '
CONST VT.RECORD = 10                '
CONST VT.OBJECT = 11                '
CONST VT.ANY = 12                   '

' VEE variable shapes: VS_          '
CONST VS.UNKNOWN = 0                '
CONST VS.SCALAR = 1                 '
CONST VS.ARRAY1D = 2                '
CONST VS.ARRAY2D = 3                '
CONST VS.ARRAY3D = 4                '
CONST VS.ARRAY = 5                  '
CONST VS.ANY = 6                    '

' Function directions: in & out.
CONST PT.IN = 0
CONST PT.OUT = 1

TYPE PARAM                          'FunctionParams: fp.
  strName AS STRING * 20            'Name of input parameter.
  iType AS INTEGER                  'Type of input parameter.
  iShape AS INTEGER                 'Shape of input parameter.
  bNameClipped AS INTEGER           'TRUE if param name was truncated.
END TYPE

TYPE FUNC                           'Functions: fc.
  strName AS STRING * 50            'Function name.
  iNumInputs AS INTEGER             'Number of input parameters.
  piInputs AS INTEGER               'Index of input params defs.
  iNumOutputs AS INTEGER            'Number of output parameters.
  piOutputs AS INTEGER              'Index of output params defs.
  bNameClipped AS INTEGER           'TRUE if func name was truncated.
END TYPE

' General purpose globals
DIM SHARED giNextFunc AS INTEGER    'Next function struct to alloc.
DIM SHARED giNextParam AS INTEGER   'Next parameter struct to alloc.

DIM SHARED strInFile AS STRING      'Input file name.
DIM SHARED strOutFile AS STRING     'Output file name.
DIM SHARED strNormFile AS STRING    'Normalized output file name.

' Master function & parameter def arrays.
DIM SHARED fcFuncs(1 TO MAX.FUNCS) AS FUNC
DIM SHARED fpParams(1 TO MAX.FUNCS * MAX.PARAMSPERFUNC) AS PARAM

CLS
giNextFunc = 1
giNextParam = 1

INPUT "Input file      :"; strInFile
'INPUT "Normalized file :"; strNormFile
INPUT "Output file     :"; strOutFile

'PRINT "Normalizing input file...  ";
'NormalizeInputFile
PRINT

PRINT "Anal-izing function data...";
AnalyzeFunctionData
PRINT

PRINT "Writing output file...";
WriteOutputFile
PRINT

END

FUNCTION AllocFunc% (strNameData AS STRING)

  DIM iQuote1 AS INTEGER
  DIM iQuote2 AS INTEGER
  DIM iThisFunc AS INTEGER

  DIM strFuncName AS STRING
 
  ' Check for function overflow.
  IF giNextFunc > MAX.FUNCS THEN
    AllocFunc% = 0
    EXIT FUNCTION
  END IF

  ' Get name.
  iThisFunc = giNextFunc
  giNextFunc = giNextFunc + 1
  iQuote1 = INSTR(strNameData, CHR$(34))
  iQuote2 = INSTR(iQuote1 + 1, strNameData, CHR$(34))
  strFuncName = MID$(strNameData, iQuote1 + 1, iQuote2 - iQuote1 - 1)
 
  IF LEN(strFuncName) > MAX.FUNCNAME THEN
    fcFuncs(iThisFunc).bNameClipped = TRUE
    strFuncName = LEFT$(strFuncName, MAX.FUNCNAME)
  END IF

  fcFuncs(iThisFunc).strName = strFuncName
  AllocFunc% = iThisFunc

END FUNCTION

FUNCTION AllocParam% (strNameData AS STRING)
 
  DIM iQuote1 AS INTEGER
  DIM iQuote2 AS INTEGER
  DIM iThisParam AS INTEGER

  DIM strParamName AS STRING

  ' Check for param overflow.
  IF giNextParam > MAX.FUNCS * MAX.PARAMSPERFUNC THEN
    AllocParam% = 0
    EXIT FUNCTION
  END IF

  ' Get name.
  iThisParam = giNextParam
  giNextParam = giNextParam + 1
  iQuote1 = INSTR(strNameData, CHR$(34))
  iQuote2 = INSTR(iQuote1 + 1, strNameData, CHR$(34))
  strParamName = MID$(strNameData, iQuote1 + 1, iQuote2 - iQuote1 - 1)

  IF LEN(strParamName) > MAX.PARAMNAME THEN
    fcFuncs(iThisParam).bNameClipped = TRUE
    strParamName = LEFT$(strParamName, MAX.PARAMNAME)
  END IF

  fpParams(iThisParam).strName = strParamName
  AllocParam% = iThisParam

END FUNCTION

SUB AnalyzeFunctionData

  DIM bInFuncs AS INTEGER           'If TRUE, processing functions.

  DIM strInLine AS STRING           'Input buffer.

  bInFuncs = FALSE
  OPEN strInFile FOR INPUT AS #1

  DO WHILE NOT EOF(1)
    INPUT #1, strInLine
    IF strInLine = "(UserFunctions" THEN
      bInFuncs = TRUE
      EXIT DO
    END IF
  LOOP
  IF NOT bInFuncs THEN EXIT SUB

  DO WHILE NOT EOF(1)
    INPUT #1, strInLine
    IF LEFT$(strInLine, 8) = "(context" AND strInLine <> "(contextCarrier" THEN
      IF GetFunctionInfo = FALSE THEN EXIT SUB
    END IF
  LOOP

  CLOSE #1

END SUB

FUNCTION CountRightParens% (strLine AS STRING)
  DIM i AS INTEGER
  DIM iNumParens AS INTEGER

  iNumParens = 0
  FOR i = 1 TO LEN(strLine)
    IF MID$(strLine, i, 1) = ")" THEN iNumParens = iNumParens + 1
  NEXT i
  CountRightParens% = iNumParens
END FUNCTION

FUNCTION GetFunctionInfo%

  DIM iThisFunc AS INTEGER
  DIM strInLine AS STRING

  ' Look for name.
  DO WHILE NOT EOF(1)
    INPUT #1, strInLine
    IF LEFT$(strInLine, 5) = "(name" THEN
      bGotName = TRUE
      EXIT DO
    END IF
  LOOP
  IF NOT bGotName THEN
    GetFunctionInfo% = FALSE
    EXIT FUNCTION
  END IF

  ' Get function info struct.
  iThisFunc = AllocFunc(strInLine)
  IF iThisFunc = 0 THEN
    GetFunctionInfo% = FALSE
    EXIT FUNCTION
  END IF

  ' Get inputs & outputs.
  DO WHILE NOT EOF(1)
    INPUT #1, strInLine
   
    IF LEFT$(strInLine, 6) = "(input" THEN
      IF fcFuncs(iThisFunc).iNumInputs = 0 THEN fcFuncs(iThisFunc).piInputs = giNextParam
      fcFuncs(iThisFunc).iNumInputs = fcFuncs(iThisFunc).iNumInputs + 1
      IF GetFunctionParam(PT.IN) = 0 THEN EXIT DO
    END IF

    IF LEFT$(strInLine, 7) = "(output" THEN
      IF fcFuncs(iThisFunc).iNumOutputs = 0 THEN fcFuncs(iThisFunc).piOutputs = giNextParam
      fcFuncs(iThisFunc).iNumOutputs = fcFuncs(iThisFunc).iNumOutputs + 1
      IF GetFunctionParam(PT.OUT) = 0 THEN EXIT DO
    END IF

    IF LEFT$(strInLine, 11) = "(deviceList" THEN EXIT DO
  LOOP

  GetFunctionInfo% = TRUE

END FUNCTION

FUNCTION GetFunctionParam% (iDirection AS INTEGER)

  DIM iThisParam AS INTEGER

  DIM strType AS STRING
  DIM strShape AS STRING
  DIM strInLine AS STRING

  DIM fp AS PARAM

  ' Get name & alloc.
  WHILE LEFT$(strInLine, 5) <> "(name"
    INPUT #1, strInLine
    IF strInLine = "(deviceList" THEN
      GetFunctionParam% = 0
      EXIT FUNCTION
    END IF
  WEND

  iThisParam = AllocParam(strInLine)
  IF iThisParam = 0 THEN
    GetFunctionParam% = 0
    EXIT FUNCTION
  END IF

  ' If this is an out param, quit here.
  IF iDirection = PT.OUT THEN
    GetFunctionParam% = iThisParam
    EXIT FUNCTION
  END IF
 
  ' Get & decode data type.
  fp = fpParams(iThisParam)
  WHILE LEFT$(strInLine, 9) <> "(datatype"
    INPUT #1, strInLine
    IF strInLine = "(deviceList" THEN
      GetFunctionParam% = 0
      EXIT FUNCTION
    END IF
  WEND

  iTypStart = INSTR(strInLine, " ") + 1
  iTypEnd = INSTR(strInLine, ")")
  strType = MID$(strInLine, iTypStart, iTypEnd - iTypStart)

  SELECT CASE strType
    CASE "Int32"
      fp.iType = VT.INT32
    CASE "Real"
      fp.iType = VT.REAL
    CASE "PComplex"
      fp.iType = VT.PCOMPLEX
    CASE "Complex"
      fp.iType = VT.COMPLEX
    CASE "Waveform"
      fp.iType = VT.WAVEFORM
    CASE "Spectrum"
      fp.iType = VT.SPECTRUM
    CASE "Coord"
      fp.iType = VT.COORD
    CASE "Text"
      fp.iType = VT.TEXT
    CASE "Enum"
      fp.iType = VT.ENUM
    CASE "Record"
      fp.iType = VT.RECORD
    CASE "Object"
      fp.iType = VT.OBJECT
    CASE "Any"
      fp.iType = VT.ANY
    CASE ELSE
      fp.iType = VT.UNKNOWN
  END SELECT

  ' Get & decode shape.
  WHILE LEFT$(strInLine, 6) <> "(shape"
    INPUT #1, strInLine
    IF strInLine = "(deviceList" THEN
      GetFunctionParam% = 0
      EXIT FUNCTION
    END IF
  WEND

  iQuote1 = INSTR(strInLine, CHR$(34))
  iQuote2 = INSTR(iQuote1 + 1, strInLine, CHR$(34))
  strShape = MID$(strInLine, iQuote1 + 1, iQuote2 - iQuote1 - 1)
  SELECT CASE strShape
    CASE "Scalar"
      fp.iShape = VS.SCALAR
    CASE "Array 1D"
      fp.iShape = VS.ARRAY1D
    CASE "Array 2D"
      fp.iShape = VS.ARRAY2D
    CASE "Array 3D"
      fp.iShape = VS.ARRAY3D
    CASE "Array"
      fp.iShape = VS.ARRAY
    CASE "Any"
      fp.iShape = VS.ANY
    CASE ELSE
      fp.iShape = VS.UNKNOWN
  END SELECT

  fpParams(iThisParam) = fp
  GetFunctionParam% = iThisParam
END FUNCTION

FUNCTION GetParamShape$ (iShape AS INTEGER)
 
  SELECT CASE iShape
    CASE VS.UNKNOWN
      GetParamShape$ = "Unknown"
    CASE VS.SCALAR
      GetParamShape$ = "Scalar"
    CASE VS.ARRAY1D
      GetParamShape$ = "Array 1D"
    CASE VS.ARRAY2D
      GetParamShape$ = "Array 2D"
    CASE VS.ARRAY3D
      GetParamShape$ = "Array 3D"
    CASE VS.ARRAY
      GetParamShape$ = "Array"
    CASE VS.ANY
      GetParamShape$ = "Any"
  END SELECT

END FUNCTION

FUNCTION GetParamType$ (iType AS INTEGER)

  SELECT CASE iType
    CASE VT.UNKNOWN
      GetParamType$ = "Unknown"
    CASE VT.INT32
      GetParamType$ = "INT32"
    CASE VT.REAL
      GetParamType$ = "REAL"
    CASE VT.PCOMPLEX
      GetParamType$ = "PCOMPLEX"
    CASE VT.COMPLEX
      GetParamType$ = "COMPLEX"
    CASE VT.WAVEFORM
      GetParamType$ = "WAVEFORM"
    CASE VT.SPECTRUM
      GetParamType$ = "SPECTRUM"
    CASE VT.COORD
      GetParamType$ = "COORD"
    CASE VT.TEXT
      GetParamType$ = "TEXT"
    CASE VT.ENUM
      GetParamType$ = "ENUM"
    CASE VT.RECORD
      GetParamType$ = "RECORD"
    CASE VT.OBJECT
      GetParamType$ = "OBJECT"
    CASE VT.ANY
      GetParamType$ = "ANY"
    CASE ELSE
      GetParamType$ = "illegal type"
  END SELECT

END FUNCTION

SUB ListData

  DIM i AS INTEGER
  DIM j AS INTEGER
  DIM iOffset AS INTEGER

  OPEN "Data.txt" FOR OUTPUT AS #1

  FOR i = 1 TO giNextFunc - 1
    PRINT #1, STRING$(78, "-")
    PRINT #1, RTRIM$(fcFuncs(i).strName),
    PRINT #1, fcFuncs(i).iNumInputs; " ("; fcFuncs(i).piInputs; ")",
    PRINT #1, fcFuncs(i).iNumOutputs; " ("; fcFuncs(i).piOutputs; ")"
    PRINT #1,

    iOffset = fcFuncs(i).piInputs
    FOR j = 0 TO fcFuncs(i).iNumInputs - 1
      PRINT #1, RTRIM$(fpParams(iOffset + j).strName),
      PRINT #1, fpParams(iOffset + j).iType; " ";
      PRINT #1, fpParams(iOffset + j).iShape
    NEXT j
    PRINT #1,

    iOffset = fcFuncs(i).piOutputs
    FOR j = 0 TO fcFuncs(i).iNumOutputs - 1
      PRINT #1, RTRIM$(fpParams(iOffset + j).strName),
      PRINT #1, fpParams(iOffset + i).iType; " ";
      PRINT #1, fpParams(iOffset + i).iShape
    NEXT j
    PRINT #1,

  NEXT i
  CLOSE #1

END SUB

SUB NormalizeInputFile

  DIM i AS INTEGER               'GP loop var.
  DIM iLevel AS INTEGER          'Indentation level.
  DIM iNumRightParens AS INTEGER 'Anti-Indentation level on current line.

  DIM strInLine AS STRING        'Input line buffer.

  iLevel = 0
  OPEN strInFile FOR INPUT AS #1
  OPEN strNormFile FOR OUTPUT AS #2

  WHILE NOT EOF(1)
    INPUT #1, strInLine
    PRINT #2, STRING$(iLevel, "  ");
    IF INSTR(strInLine, "(") THEN iLevel = iLevel + 1
    IF INSTR(strInLine, ")") THEN
      iNumRightParens = CountRightParens(strInLine)
      PRINT #2, LEFT$(strInLine, LEN(strInLine) - iNumRightParens)
      FOR i = 1 TO iNumRightParens
        IF iLevel > 0 THEN iLevel = iLevel - 1
        PRINT #2, STRING$(iLevel, "  ");
        PRINT #2, ")"
      NEXT i
    ELSE
      PRINT #2, strInLine
    END IF
  WEND
 
  CLOSE #2
  CLOSE #1

END SUB

SUB WriteOutputFile

  DIM iFunc AS INTEGER
  DIM iParam AS INTEGER
  DIM iOffset AS INTEGER
  DIM iNameLen AS INTEGER
  DIM iMaxNameLen AS INTEGER

  DIM strOutput AS STRING
  DIM strSepLine AS STRING
  DIM strParamName AS STRING

  strSepLine = STRING$(78, "-")

  OPEN strOutFile FOR OUTPUT AS #1
 
  ' Title
  PRINT #1, "GENDOC listing"
  PRINT #1, giNextFunc - 1; " Functions, "; giNextParam - 1; " Parameters."
  PRINT #1,

  FOR iFunc = 1 TO giNextFunc - 1
    iMaxNameLen = 0
    PRINT #1, strSepLine
   
    ' Output type.
    IF fcFuncs(iFunc).iNumOutputs = 0 THEN
      PRINT #1, "VOID ";
    ELSE
      iOffset = fcFuncs(iFunc).piOutputs
      iNameLen = LEN(RTRIM$(fpParams(iOffset).strName))
      IF iNameLen > iMaxNameLen THEN iMaxNameLen = iNameLen
      PRINT #1, GetParamType$(fpParams(iOffset).iType); " ";
    END IF

    ' Test the rest of the outputs for length.
    FOR iParam = 1 TO fcFuncs(iFunc).iNumOutputs - 1
      iNameLen = LEN(RTRIM$(fpParams(iOffset + iParam).strName))
      IF iNameLen > iMaxNameLen THEN iMaxNameLen = iNameLen
    NEXT iParam

    ' Function name.
    PRINT #1, RTRIM$(fcFuncs(iFunc).strName); "(";

    ' Input parameter list.
    IF fcFuncs(iFunc).iNumInputs > 0 THEN
      iOffset = fcFuncs(iFunc).piInputs
      FOR iParam = 0 TO fcFuncs(iFunc).iNumInputs - 1
        strParamName = RTRIM$(fpParams(iOffset + iParam).strName)
        iNameLen = LEN(strParamName)
        IF iNameLen > iMaxNameLen THEN iMaxNameLen = iNameLen
        PRINT #1, GetParamType$(fpParams(iOffset + iParam).iType); " "; strParamName;
        IF iParam < fcFuncs(iFunc).iNumInputs - 1 THEN PRINT #1, ", ";
      NEXT iParam
    END IF
    PRINT #1, ");"
    PRINT #1,

    ' Hike max name len to account for formatting
    iMaxNameLen = iMaxNameLen + 2
   
    ' Inputs section.
    iOffset = fcFuncs(iFunc).piInputs
    PRINT #1, "Inputs:"
    FOR iParam = 0 TO fcFuncs(iFunc).iNumInputs - 1
      strOutput = "  " + RTRIM$(fpParams(iOffset + iParam).strName) + STRING$(iMaxNameLen, " ")
      PRINT #1, LEFT$(strOutput, iMaxNameLen); " - [";
      PRINT #1, GetParamType$(fpParams(iOffset + iParam).iType); ", ";
      PRINT #1, GetParamShape$(fpParams(iOffset + iParam).iShape); "]"
    NEXT iParam
    PRINT #1,

    ' Outputs section.
    iOffset = fcFuncs(iFunc).piOutputs
    PRINT #1, "Outputs:"
    FOR iParam = 0 TO fcFuncs(iFunc).iNumOutputs - 1
      strOutput = "  " + RTRIM$(fpParams(iOffset + iParam).strName) + STRING$(iMaxNameLen, " ")
      PRINT #1, LEFT$(strOutput, iMaxNameLen); " - ";
      PRINT #1, "["; GetParamType$(fpParams(iOffset + iParam).iType); "]"
    NEXT iParam
    PRINT #1,
 
  NEXT iFunc
  CLOSE #1

END SUB

