//-*- mode: c++;-*-

//////////////////////////////////////////////////////////////////////
// $RCSfile: SparmServerPNA.cs,v $
// $Revision: 1.16 $
// $Date: 2003/10/23 20:34:43 $
// $Author: doomer $
//
// Copyright (c) 2002, Agilent Technologies.  All rights reserved.
//////////////////////////////////////////////////////////////////////

using System;
using Agilent.TMFramework;
using Agilent.TMFramework.InstrumentIO;

namespace Agilent.MeasurementServer.SparmMsmt
{
	/// <summary>
	/// Agilent375xSparmServer class includes basic functions of two
	/// ports power depentent (power swept) S-parameter measurement
	/// and three ports single (power) point three ports S-parameter
	/// measurement for Agilent E88xxA, E83xxA, and N338xA PNA.
	/// </summary>
	public class Agilent835xSparmServer
	{
		DirectIO pnaIO;

		int m_chan = 1;
		string[] m_strMeas = new string[16];
        //string[] m_strSparam = {"S11", "S12", "S13", "S21", "S22", "S23", "S31", "S32", "S33", "S14", "S24", "S34", "S41", "S42", "S43", "S44"};
        //string[] m_strLoad   = { ",2",    "",    "",  "",    ",1",    "",    "",    "",  ",1", "",    "",    "",    "",    "",    "",    ",1"};

        string[] m_strSparam = {"S11", "S12", "S13", "S14", "S21", "S22", "S23", "S24", "S31", "S32", "S33", "S34", "S41", "S42", "S43", "S44"};
        string[] m_strLoad   = { ",2",    "",    "",    "",    "",  ",1",    "",    "",    "",    "",  ",1",    "",    "",    "",    "",  ",1"};
        
		bool m_pnaClosed = true;

		string m_strFormData = null;
		string m_strTrigSour = null;
		string m_strSourPow1AttAuto = null;
		string m_strSourPow2AttAuto = null;
		string m_strSourPowCoup = null;
		string m_strSourPow1 = null;
		string m_strSourPow2 = null;
		string m_strSensFreqStar = null;
		string m_strSensFreqStop = null;
		string m_strSensSwePoin = null;
		string m_trigMode = null;
		int m_groupCount = 1;
		bool m_forceMsmt = false;

		enum InstrumentType {
		  PNA,
		  ENA_A,
		  ENA_B,
		  UNSUPPORTED
		};

		/// Field to hold PNA, ENA_A, ENA_B
		InstrumentType VNA_Type = InstrumentType.UNSUPPORTED; 

		const double PNAmin = 1e-10;
		const int PNAtimeout = 1000;		// millisecond

		/// <summary>
		/// Initial communication to instrument.  Instrument may be
		/// reset according to the input parameter.  According to
		/// the input parameter, instrument may also be asked for ID
		/// to check if it is supported.
		/// </summary>
		/// <param name="visaResourceId">The string contains a VISA
		/// id that identify the instrument.</param>
		/// <param name="idQuery">If 'true', instrument ID will be
		/// checked and an exception will be thrown if the instrument
		/// is not supported.</param>
		/// <param name="preset">If 'true', instrument will be reset
		/// before the further measurement.</param>
		public Agilent835xSparmServer(string visaResourceId, bool idQuery, bool preset)
		{
            // System.Diagnostics.Debugger.Break();
            
			try
			{
				pnaIO = new Agilent.TMFramework.InstrumentIO.DirectIO( visaResourceId, true, PNAtimeout );

				m_pnaClosed = false;
				
				/// Probably never want to preset
				if(preset)
				{
					pnaIO.Write("*RST;");
				}

				if(idQuery) {
				  pnaIO.Write("*IDN?;");
				  string retID = pnaIO.Read();
				  Console.WriteLine(retID);

				  if (retID.IndexOf("Agilent")>=0) {
				    if ((retID.IndexOf("E88")>=0) || 
					(retID.IndexOf("E83")>=0) || 
					(retID.IndexOf("N338")>=0)) {
				      VNA_Type = InstrumentType.PNA; 
				    }
				    else if ((retID.IndexOf("E5070A")>=0) || 
					     (retID.IndexOf("E5071A")>=0)) {
				      // We'll add support for ENA A in 2003C if time permits.  
				      // if we don't add in 2003C, we'll wait for customer requests.  Customer 
				      // requests have only been for ENA B
				      // VNA_Type = InstrumentType.ENA_A;
				      VNA_Type = InstrumentType.UNSUPPORTED;
				    }
				    else if ((retID.IndexOf("E5070B")>=0) || 
					     (retID.IndexOf("E5071B")>=0)) {
				      VNA_Type = InstrumentType.ENA_B;
				    }
				    else {
				      VNA_Type = InstrumentType.UNSUPPORTED;
				    }
				  }
				  if (VNA_Type == InstrumentType.UNSUPPORTED) {
				    m_pnaClosed = true;
				    pnaIO.Close();
				    throw new Exception("Unsupported Instrument: " + retID);
				  }
				}
			}
			catch(VisaException v)
			{
				m_pnaClosed = true;

				if(pnaIO != null)
				{
					pnaIO.Close();
				}
				
				throw new ArgumentException("wrong visaID: " + v.Message, visaResourceId);
			}

			int i;

			for (i=0; i<4; i++)
				m_strMeas[i] = "BAS_" + m_strSparam[i];

			for (i=4; i<16; i++)
				m_strMeas[i] = "EXT_" + m_strSparam[i];
		}

		/// <summary>
		/// Setup PNA for required S-parameter measurement, including,
		/// couple/uncouple port1 and port2 power;
		/// turn off port1 and port2 auto attenuator mode;
		/// manual triggering;
		/// call to set/get frequency;
		/// call to define requirement S-parameter measurement.
		/// </summary>
		/// <param name="powCouple">'ture' to couple port1 and port2
		/// power; 'false' to uncouple port1 and port2 power</param>
		/// <param name="currentFreq">'ture' to set the PNA frequency
		/// and number of points of sweep; 'false' to get the current
		/// PNA settings.</param>
		/// <param name="startFreq">Set the PNA start frequency. If
		/// 'currentFreq' is 'false', return the current instrument
		/// setting.</param>
		/// <param name="stopFreq">Set the PNA stop frequency. If
		/// 'currentFreq' is 'false', return the current instrument
		/// setting.</param>
		/// <param name="points">Set the PNA number of points of sweep.
		/// If 'currentFreq' is 'false', return the current instrument
		/// setting.</param>
		/// <param name="s">An array defines which S-parameter will be
		/// measured.</param>
		public void Setup(bool powCouple, bool currentFreq, ref double startFreq, ref double stopFreq, ref int points, bool[] s)
		{
			if (m_pnaClosed) return;

			try
			{
				// keep the current data format, then
				// setup data format of S-parameter as double
				pnaIO.Write("FORM:DATA?");
				m_strFormData = pnaIO.Read();

				// PNA can be set to real 32 to be consistent with ENA
				// I've left it the same as 
				if (VNA_Type == InstrumentType.PNA) {
				  pnaIO.Write("FORM:DATA REAL, 64");
				}
				else {
				  // ENA_A or ENA_B
				  pnaIO.Write("FORM:DATA REAL");
				}

				// keep the current sweep mode, then
				// setup sweep to manual trigger
				pnaIO.Write("TRIG:SOUR?");
				m_strTrigSour = pnaIO.Read();

				if (VNA_Type == InstrumentType.PNA) {
					pnaIO.WriteLine(":sens" + m_chan + ":swe:mode?");
					m_trigMode = pnaIO.Read(true);
					pnaIO.WriteLine(":sens" + m_chan + ":swe:gro:coun?");
					m_groupCount = pnaIO.ReadNumberAsInt32();

					pnaIO.WriteLine(":sens" + m_chan + ":swe:mode hold");
					
				  // keep the current port1 power
				  pnaIO.Write("SOUR" + m_chan + ":POW1?");
				  m_strSourPow1 = pnaIO.Read();
				  
				  // keep the current port2 power
				  pnaIO.Write("SOUR" + m_chan + ":POW2?");
				  m_strSourPow2 = pnaIO.Read();

				  // ENA does not support attenuator
				  // keep the current port1 auto attenuator mode
				  pnaIO.Write("SOUR" + m_chan + ":POW1:ATT:AUTO?");
				  m_strSourPow1AttAuto = pnaIO.Read();
				  
				  // keep the current port2 auto attenuator mode
				  pnaIO.Write("SOUR" + m_chan + ":POW2:ATT:AUTO?");
				  m_strSourPow2AttAuto = pnaIO.Read();

				}
				else {
				  // ENA_A or ENA_B
				  // make sure that m_chan is correct?
				  pnaIO.Write("init" + m_chan + ":cont on");
				  pnaIO.Write("trig:sour bus;");

				  // FIXME - this only works for ENA_B not ENA_A
				  // keep the current port1 power
				  pnaIO.Write("SOUR" + m_chan + ":POW:PORT1?");
				  m_strSourPow1 = pnaIO.Read();
				  
				  // keep the current port2 power
				  pnaIO.Write("SOUR" + m_chan + ":POW:PORT2?");
				  m_strSourPow2 = pnaIO.Read();
				}

				
				// FIX_ME: ENA_A does not support two different powers at port 1 and 2,
				//This needs to be caught higher up, and not send different port 1 and port2
				//powers.  ENA_B can support both.
				// un-couple port1 and port2, and disable auto attenuator control

				if (VNA_Type == InstrumentType.PNA) {

				  // keep the current port1&2 couple mode
				  pnaIO.Write("SOUR" + m_chan + ":POW:COUP?");
				  m_strSourPowCoup = pnaIO.Read();

				  if (powCouple) 
				    pnaIO.Write("SOUR" + m_chan + ":POW:COUP ON");
				  else {
				    pnaIO.Write("SOUR" + m_chan + ":POW:COUP OFF");
				    // Should be sending this for the PNA?, auto OK?  
				    // Deactivating for 2003C - Joel D review - the UI is designed to 
				    // assume customer setup instrument correctly
				    // pnaIO.Write("SOUR" + m_chan + ":POW2:ATT:AUTO OFF");
				  }

				  // Deactivating for 2003C - Joel D review - the UI is designed to 
				  // assume customer setup instrument correctly
				  // pnaIO.Write("SOUR" + m_chan + ":POW1:ATT:AUTO OFF");

				}
				else { // ENA_A & ENA_B

				  // keep the current port1&2 couple mode
				  // FIXME - Update for ENA_A
				  pnaIO.Write("SOUR" + m_chan + ":POW:PORT:COUP?");
				  m_strSourPowCoup = pnaIO.Read();

				  // ENA_B
				  if (powCouple)
				    pnaIO.Write("SOUR" + m_chan + ":POW:PORT:COUP ON");
				  else
				    pnaIO.Write("SOUR" + m_chan + ":POW:PORT:COUP OFF");
				}

				if (currentFreq)
					GetFrequency(out startFreq, out stopFreq, out points);
				else
					SetFrequency(ref startFreq, ref stopFreq, points);
				
				if (VNA_Type == InstrumentType.PNA) 
				  CreateMeasurement(s);
				else
				  CreateMeasurementENA(s);
			}
			catch(VisaException v)
			{
				m_pnaClosed = true;
				pnaIO.Close();
				throw new Exception("Can not setup measurement: " + v.Message);
			}
		}

		/// <summary>
		/// Used for behavioral modeling.  If the instrument is in hold mode, we need to
		/// force it to take a new sweep.  Normally in hold mode, we would just return the data currently
		/// displayed.
		/// </summary>
		public void ForceMeasurement()
		{
			m_forceMsmt = true;
			TriggerMeasurement();
		}

		/// <summary>
		/// Trigger PNA to make a single sweep
		/// </summary>
		public void TriggerMeasurement()
		{
			if (m_pnaClosed) return;

			try
			{
				if (VNA_Type == InstrumentType.PNA) {
					if(m_forceMsmt)
					{
						pnaIO.WriteLine(":sens" + m_chan + ":swe:gro:coun 1");
						pnaIO.WriteLine(":sens" + m_chan + ":swe:mode gro");
					}
					else 
					{
						if(m_trigMode != "HOLD")
						{
							pnaIO.WriteLine(":sens" + m_chan + ":swe:gro:coun 1");
							pnaIO.WriteLine(":sens" + m_chan + ":swe:mode gro");
						}
					}
				}
				else {
				  //ENA needs to be sent "trig:sing" then "*opc?"
				  pnaIO.Write(":trig:sing");
				}

				pnaIO.Write("*OPC?");
				
				int oldTimeOut = pnaIO.Timeout;
				pnaIO.Timeout = 60 * 1000;

				int success = pnaIO.ReadNumberAsInt32();		// should get +1
				if (success != 1)
				{
					Release();
					throw new Exception("Measurement could not complete");
				}

				pnaIO.Timeout = oldTimeOut;
			}
			catch(VisaException v)
			{
				m_pnaClosed = true;
				pnaIO.Close();
				throw new Exception("Can not trigger measurement: " + v.Message);
			}

			return;
		}

		/// <summary>
		/// Get measurement result for specified S-parameter 
		/// </summary>
		/// <param name="paramIndex">
		/// 0 - S11, 1 - S12, 2 - S21
		/// 3 - S22, 4 - S13, 5 - S23
		/// 6 - S31, 7 - S32, 8 - S33</param>
		/// <returns>(Real, imaginary) S-parameter array</returns>
		public double[] ReadSparameter(int paramIndex)
		{
			int oldTimeOut = pnaIO.Timeout;
			pnaIO.Timeout = 60 * 1000;

			if (m_pnaClosed) return null;

			try {
			  if (VNA_Type == InstrumentType.PNA) {
			    pnaIO.Write("CALC" + m_chan + ":PAR:SEL " + m_strMeas[paramIndex]);
			    pnaIO.Write("CALC" + m_chan + ":DATA? SDATA");
			    double[] S = pnaIO.ReadIeeeBlockAsDoubleArray();
			    return S;
			  }
			  else {
			    //for the ENA, the par has a number associated with parameter,
			    // send "calc"+m_chan+":par"+(paramIndex+1)+":sel" to select the correct parameter
			    pnaIO.Write("CALC" + m_chan + ":PAR"+(paramIndex+1)+":SEL ");
				
			    // for the ENA, "CALC" + m_chan + ":DATA SDAT?"
			    // remember to read back as 32 bit instead of 64
			    pnaIO.Write("CALC" + m_chan + ":DATA:SDAT?");
				double[] S = pnaIO.ReadIeeeBlockAsDoubleArray();

                pnaIO.Timeout = oldTimeOut;
                
			    return S;
			  }
			}
			catch(VisaException v)
			{
				m_pnaClosed = true;
				pnaIO.Close();
				throw new Exception("Can not get measurement result: " + v.Message);
			}
		}

		/// <summary>
		/// Get current PNA frequency settings
		/// </summary>
		/// <param name="start">Current PNA start frequency</param>
		/// <param name="stop">Current PNA stop frequency</param>
		/// <param name="points">Current number of points of sweep</param>
		public void GetFrequency(out double start, out double stop, out int points) {

		  // FIXME: for ENA and PNA, support also log type and segement sweep
		  // get the type using "SENS" + m_chan + ":swe:type?"//look for segm, log or lin.  
		  // if log, create a log ramp for frequency
		  // if segment, get the segment count using:
		  // "sens"+m_chan+ "segm:coun?",
		  // then for each segment (seg_I)
		  //	read in "sens"+m_chan+ "segm"+seg_I + ":swe:poin?"
		  //		"sens"+m_chan+ "segm"+seg_I + ":swe:star?"
		  //		"sens"+m_chan+ "segm"+seg_I + ":swe:stop?"
		  // and build a complete list

		  pnaIO.Write("SENS" + m_chan + ":swe:type?");
		  string sweepType = pnaIO.Read();

		  if (sweepType.IndexOf("LIN")<0) {
		    m_pnaClosed = true;
		    pnaIO.Close();
		    throw new Exception("Only linear frequency sweeps are currently supported, the instrument is currently set to " + sweepType);
		  }

		  start = stop = -1.0;
		  points = -1;
			
			if (m_pnaClosed) return;
			
			try 
			{
				pnaIO.Write("SENS" + m_chan + ":FREQ:STAR?");
				start = pnaIO.ReadNumberAsDouble();

				pnaIO.Write("SENS" + m_chan + ":FREQ:STOP?");
				stop = pnaIO.ReadNumberAsDouble();

				pnaIO.Write("SENS" + m_chan + ":SWE:POIN?");
				points = pnaIO.ReadNumberAsInt32();
			}
			catch(VisaException v)
			{
				m_pnaClosed = true;
				pnaIO.Close();
				throw new Exception("Can not get Frequency and SweepPoint setting: " + v.Message);
			}
		}

		/// <summary>
		/// Set PNA sweet power for port1 and port2
		/// </summary>
		/// <param name="power">Port1 power</param>
		/// <param name="powerOffset">(Port2 power) - (Port1 power)</param>
		public void SetPower(double power, double powerOffset)
		{
			if (m_pnaClosed) return;

			try
			{
				if (powerOffset == 0) {
				  if (VNA_Type == InstrumentType.PNA) {
				    pnaIO.Write("SOUR" + m_chan + ":POW1 " + power);
				  }
				  else if (VNA_Type == InstrumentType.ENA_A) {
				    // same as above for power change
				    //ENA_A needs just POW,
				    pnaIO.Write(":SOUR" + m_chan + ":POW " + power);
				  }
				  else if (VNA_Type == InstrumentType.ENA_B) {
				    //ENA_B needs POW:PORT1 or POW:PORT2
				    pnaIO.Write(":SOUR" + m_chan + ":POW:PORT1 " + power);
				  }
				}
				else {
				  if (VNA_Type == InstrumentType.PNA) {
				    pnaIO.Write("SOUR" + m_chan + ":POW1 " + power);
				    pnaIO.Write("SOUR" + m_chan + ":POW2 " + (power + powerOffset));
				  }
				  else {
				    // ENA_B support only
				    pnaIO.Write("SOUR" + m_chan + ":POW:PORT1 " + power);
				    pnaIO.Write("SOUR" + m_chan + ":POW:PORT2 " + (power + powerOffset));
				  }
				}

				if (VNA_Type == InstrumentType.PNA) {
				  pnaIO.Write("SOUR" + m_chan + ":POW1?");
				}
				else {
				  // FIXME - only ENA_B
				  pnaIO.Write("SOUR" + m_chan + ":POW:PORT1?");
				}

				double p1 = pnaIO.ReadNumberAsDouble();

				if (Math.Abs(p1 - power) > PNAmin)
				{
					Release();
					throw new ArgumentOutOfRangeException("Port1 Power", power + " dBm", "Out of range");
				}

				if (VNA_Type == InstrumentType.PNA) {
				  pnaIO.Write("SOUR" + m_chan + ":POW2?");
				}
				else {
				  // FIXME - only ENA_B
				  pnaIO.Write("SOUR" + m_chan + ":POW:PORT2?");
				}

				double p2 = pnaIO.ReadNumberAsDouble();

				if (Math.Abs(p2 - power - powerOffset) > PNAmin)
				{
					Release();
					throw new ArgumentOutOfRangeException("Port2 Power", (power + powerOffset) + " dBm", "Out of range");
				}				
			}
			catch(VisaException v)
			{
				m_pnaClosed = true;
				pnaIO.Close();
				throw new Exception("Can not set Power: " + v.Message);
			}
		}

		/// <summary>
		/// Set PNA frequency
		/// </summary>
		/// <param name="start">PNA start frequency</param>
		/// <param name="stop">PNA stop frequency</param>
		/// <param name="points">PNA number of points of sweep</param>
		private void SetFrequency(ref double start, ref double stop, int points)
		{
			if (m_pnaClosed) return;

			try
			{
				pnaIO.Write("SENS" + m_chan + ":FREQ:STAR?");
				m_strSensFreqStar = pnaIO.Read();

				pnaIO.Write("SENS" + m_chan + ":FREQ:STOP?");
				m_strSensFreqStop = pnaIO.Read();

				pnaIO.Write("SENS" + m_chan + ":SWE:POIN?");
				m_strSensSwePoin = pnaIO.Read();

				// if specified frequence range is not located inside the current
				// (calibrated) frequence range, specified range will be shrunk
				if ((start < Convert.ToDouble(m_strSensFreqStar)))
					start = Convert.ToDouble(m_strSensFreqStar);
				
				if ((stop > Convert.ToDouble(m_strSensFreqStop)))
					stop = Convert.ToDouble(m_strSensFreqStop);

				pnaIO.Write("SENS" + m_chan + ":FREQ:STAR " + start);
				pnaIO.Write("SENS" + m_chan + ":FREQ:STOP " + stop);
				pnaIO.Write("SENS" + m_chan + ":SWE:POIN " + points);
			}
			catch(VisaException v)
			{
				m_pnaClosed = true;
				pnaIO.Close();
				throw new Exception("Can not set frequency: " + v.Message);
			}

			return;
		}

		/// <summary>
		/// Create (define) corresponding S-parameter measurements
		/// as specified by parameter s[]
		/// </summary>
		/// <param name="s">An array to specify required S-parameter
		/// measurements</param>
		private void CreateMeasurement(bool[] s)
		{
			if (m_pnaClosed) return;

			int numberNewMeas = 0, numberExistMeas = 0;

			for (int i=0; i<s.Length; i++)
				if (s[i]) numberNewMeas++;

			try
			{
				pnaIO.Write("CALC" + m_chan + ":PAR:CAT?");
				string strCurrentMeas = pnaIO.Read();

				string[] str_SplitMeas = strCurrentMeas.Split(',');
				numberExistMeas = str_SplitMeas.Length / 2;

				// There are only 16 total measurements allowed for each channel.
				// If there is no enough space to create all necessary new measurements,
				// some existing measurements will be deleted.
				if (numberExistMeas + numberNewMeas >16)
				{
					for (int i=2*(16-numberNewMeas); i<2*numberExistMeas; i+=2)
						pnaIO.Write("CALC" + m_chan + ":PAR:DEL " + str_SplitMeas[i]);
				}

				for (int i=0; i < s.Length; i++)
				{
					if (!s[i]) continue;

					// FIXME - do I need to do this for ENA?
					if (strCurrentMeas.IndexOf(m_strMeas[i]) >= 0)
						pnaIO.Write("CALC" + m_chan + ":PAR:DEL " + m_strMeas[i]);

					
					pnaIO.Write("CALC" + m_chan + ":PAR:DEF " + m_strMeas[i] 
						    + "," + m_strSparam[i] + m_strLoad[i]);
					pnaIO.Write("CALC" + m_chan + ":PAR:CAT?");
					strCurrentMeas = pnaIO.Read();
					if (strCurrentMeas.IndexOf(m_strMeas[i]) < 0)
					{
						Release();
						throw new ArgumentOutOfRangeException("S-parameter", m_strSparam[i], "Can not create measurement");
					}
				}
			}
			catch(VisaException v)
			{
				m_pnaClosed = true;
				pnaIO.Close();
				throw new Exception("Can not create measurement: " + v.Message);
			}
		}

		/// <summary>
		/// Create (define) corresponding S-parameter measurements
		/// as specified by parameter s[]
		/// </summary>
		/// <param name="s">An array to specify required S-parameter
		/// measurements</param>
		private void CreateMeasurementENA(bool[] s)
		{
			if (m_pnaClosed) return;

			int numberNewMeas = 0;

			for (int i=0; i<s.Length; i++)
				if (s[i]) numberNewMeas = i + 1;

			try 
			{
				pnaIO.WriteLine(":SERV:CHAN:TRAC:COUN?");
				int traceCount = pnaIO.ReadNumberAsInt32();

				if(traceCount < numberNewMeas)
				{
					System.Text.StringBuilder sb = new System.Text.StringBuilder();
					sb.Append("\nThe instrument is configured for only ");
					sb.Append(traceCount);
					sb.Append(" traces, but you have requested ");
					sb.Append(numberNewMeas);
					sb.Append(" measurements.\nPlease reconfigure the instrument to contain ");
					sb.Append(numberNewMeas);
					sb.Append(" traces.");
					throw new Exception(sb.ToString());
				}

				pnaIO.Write("CALC" + m_chan + ":PAR:COUN " + numberNewMeas);

				pnaIO.WriteLine(":syst:err?");
				string err = pnaIO.Read(true);
		  
				for (int i=0; i<s.Length; i++) 
				{
					if (!s[i]) continue;
					pnaIO.Write(":calc" + m_chan +":par" + (i+1) + ":def " + m_strSparam[i]);

					pnaIO.WriteLine(":syst:err?");
					err = pnaIO.Read(true);

					pnaIO.Write(":CALC" + m_chan + ":PAR" + (i+1) + ":DEF?");
					string strCurrentMeas = pnaIO.Read();
					if (strCurrentMeas.IndexOf(m_strSparam[i]) < 0) 
					{
						Release();
						throw new ArgumentOutOfRangeException("S-parameter", m_strSparam[i], "Can not create measurement");
					}
				}
			}
			catch(Exception e) 
			{
				m_pnaClosed = true;
				pnaIO.Close();
				throw;
			}
		}

		/// <summary>
		/// Delete measurements defined by other functions
		/// </summary>
		private void DeleteMeasurement()
		{
			if (m_pnaClosed) return;

			try
			{
				pnaIO.Write("CALC" + m_chan + ":PAR:CAT?");
				string strCurrentTraces = pnaIO.Read();

				for (int i=0; i<9; i++)
				{
					if (strCurrentTraces.IndexOf(m_strMeas[i]) >= 0)
						pnaIO.Write("CALC" + m_chan + ":PAR:DEL " + m_strMeas[i]);
				}
			}
			catch(VisaException v)
			{
				m_pnaClosed = true;
				pnaIO.Close();
				throw new Exception("Can not delete measurement: " + v.Message);
			}			

			return;
		}
		
//		~PNAWrapper()

		/// <summary>
		/// Restore the original instrument setting and release PNA
		/// </summary>
		public void Release()
		{
			if (m_pnaClosed) return;

			try
			{
			  // We do not support restoring ENA state
			  if (VNA_Type == InstrumentType.PNA) {

				DeleteMeasurement();

				if (m_strFormData != null)
					pnaIO.Write("FORM:DATA " + m_strFormData);

				if (m_strTrigSour != null)
					pnaIO.Write("TRIG:SOUR " + m_strTrigSour);

				  pnaIO.WriteLine(":sens" + m_chan + ":swe:gro:coun " + m_groupCount.ToString());

				  if (m_trigMode != null)
					  pnaIO.WriteLine(":sens" + m_chan + ":swe:mode " + m_trigMode);

				if (m_strSourPowCoup != null)
					pnaIO.Write("SOUR" + m_chan + ":POW:COUP " + m_strSourPowCoup);

				if (m_strSourPow1AttAuto != null)
					pnaIO.Write("SOUR" + m_chan + ":POW1:ATT:AUTO " + m_strSourPow1AttAuto);

				if (m_strSourPow2AttAuto != null)
					pnaIO.Write("SOUR" + m_chan + ":POW2:ATT:AUTO " + m_strSourPow2AttAuto);

				if (m_strSourPow1 != null)
					pnaIO.Write("SOUR" + m_chan + ":POW1 " + m_strSourPow1);

				if (m_strSourPow2 != null)
					pnaIO.Write("SOUR" + m_chan + ":POW2 " + m_strSourPow2);

				if (m_strSensFreqStar != null)
					pnaIO.Write("SENS" + m_chan + ":FREQ:STAR " + m_strSensFreqStar);

				if (m_strSensFreqStop != null)
					pnaIO.Write("SENS" + m_chan + ":FREQ:STOP " + m_strSensFreqStop);

				if (m_strSensSwePoin != null)
					pnaIO.Write("SENS" + m_chan + ":SWE:POIN " + m_strSensSwePoin);
			  }
			}
			catch(VisaException v)
			{
				throw new Exception("Can not restore instrument settings: " + v.Message);
			}
			finally
			{
				m_pnaClosed = true;
				pnaIO.Close();
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////
// $Log: SparmServerPNA.cs,v $
// Revision 1.16  2003/10/23 20:34:43  doomer
// adding revision information.
//
// Revision 1.15  2003/10/14 21:52:46  doomer
// fix for Instrument_Links.285.
//
// Revision 1.14  2003/10/10 20:56:05  doomer
// Fix for Instrument_Links.286.
//
// Revision 1.13  2003/08/08 17:19:16  doomer
// cleaning up the order of measurement definitions
//
// Revision 1.12  2003/08/07 22:12:01  doomer
// removing some debug code
//
// Revision 1.11  2003/08/07 22:10:33  doomer
// OK, guys, we need to quit hard coding array lengths into for loops.
//
// Revision 1.10  2003/07/28 20:13:34  doomer
// making 4-port s parameter measurements work.
//
// Revision 1.9  2003/07/25 18:53:31  doomer
// adding support for 4-port s parameters
//
// Revision 1.8  2003/07/24 19:57:01  doomer
// fix for Instrument_Links.207 & Instrument_Links.206
//
// Revision 1.7  2003/07/15 18:52:46  jpino
// first version to add support for the ENA
//
// Revision 1.6  2003/05/02 01:27:16  blin
// Specify load port for multi-port reflection measurement.
//
// Revision 1.5  2003/04/23 23:41:20  blin
// ATM Instrument_Links.137
// Prevent functions from re-entrying after instrument has been closed.
//
// Revision 1.4  2003/04/03 23:56:33  blin
// Update DirectIO openTimeout from 25 ms to 1s.
//
// Revision 1.3  2003/03/02 05:20:59  blin
// Adding cvs headers.
//
// Revision 1.2  2003/03/02 05:17:16  blin
// Updating exception handler and comments.
//
//////////////////////////////////////////////////////////////////////


