//////////////////////////////////////////////////////////////////////
/// \file QarbDiscoveryLibrary.cpp This file is the implementation of a DLL
/// that tries to identify all the N6030A instruments which reside
/// in the card cage with the computer or are attenched to the computer
/// through a PXI interface.  Visa by itself will not identify these
/// instruments, so it is necessary to use the N6030A's IVI driver to 
/// perform the discovery.  This libary is written in managed c++ and
/// is compiled into a .Net DLL, not a native Windows DLL.  The
/// implications are that we can link directly against this DLL from 
/// a C# file, like the CM server or STW Instrument Explorer, and
/// and that dynamically allocated memory is garbage collected,
/// meaning that there is no need to delete memory we have new'd.
/// For an example illustrating the use of the functions in this 
/// library, refer the file QarbDiscovery.cpp in the parent directory
/// of the directory that contains this source file.
//////////////////////////////////////////////////////////////////////
// $RCSfile: QarbDiscoveryLibrary.cpp,v $
// $Revision: 1.3 $
// $Date: 2005/11/30 22:08:59 $
// $Author: doomer $
//////////////////////////////////////////////////////////////////////

#include <windows.h>

#include "QarbDiscoveryLibrary.h"
#include "QarbDiscoveryHelper.h"

#include <visa.h>

using namespace System;
using namespace System::Text;
using namespace System::Collections;

/// Define a function pointer type we can use to call into the IVI driver's
/// instrument discovery entry point.
typedef ViStatus (_VI_FUNC *QueryQarbFn)(
    ViSession	vi,				/* always pass this as NULL; it is not used */
    ViInt16		arraySize,
    ViInt16		busNumber[],
    ViInt16		deviceNumber[],
    ViInt16		slotNumber[],
    char		*serialNumber[],
    char		*visaResourceString[],
    ViInt16		*numArbsFound );

/// This function will try to load the N6030A IVI driver in an attempt
/// to enumerate the Visa resource strings used to identify particular 
/// instruments.  We call into a native windows DLL to check that the version
/// of IVI driver on the system is late enough to meet our needs.  Earlier versions
/// of the IVI driver ued a different number of arguments in the discovery function.
/// We need a native DLL to do this, because managed C++ will not compie the version
/// information header file.
/// \param resources A reference to an ArrayList that gets populated with the Visa
/// resource strings corresponding to an instrument.  This function does not clear
/// the array; it just adds whatever resource strings it finds to the end of
/// the list.
/// \return True if we can dynamically load the IVI driver, it is of a sufficient
/// revision, the driver DLL contains the discovery function, and we succesfully
/// called the discovery function.  Otherwise, we return false.
bool QarbDiscovery::EnumQarbResources(ArrayList *resources)
{
    bool result = false;

    // rsrcIdTable is a hash table keyed off the instrument Visa resource containing
    // the inctrument description in the hash table data field.  We collect the instrument
    // description when we query for the Visa resource.  A subsequent call  to GetIdForQarbAtRsrc
    // will index into the hash table to retrieve the description.
    rsrcIdTable->Clear();

    if( ! QarbDllVersionIsSufficient()){
        return result;
    }

    HMODULE hQarbDll = LoadLibrary("AGN6030A.dll");
    if(hQarbDll){
        QueryQarbFn qFn = reinterpret_cast<QueryQarbFn>(GetProcAddress(hQarbDll,
            "AGN6030A_query_available_instruments"));
        if(qFn){
            const ViInt16 maxNumModules = 16;

            ViInt16 busNumbers[maxNumModules] = {0};
            ViInt16 deviceNumbers[maxNumModules] = {0};
            ViInt16 slotNumbers[maxNumModules] = {0};

            char *serialNumbers[maxNumModules] = {0};
            char *visaRsrcs[maxNumModules] = {0};

            ViInt16 numArbsFound = 0;

            // Calling this function introduces a memory leak.  The driver function dynamically allocates
            // space to hold the arrays of strings, but trying to free that memory causes a crash.
            // A later version of the driver DLL intends to add a function to free the allocated
            // memory.
            if(qFn(0 /* not used */, maxNumModules, busNumbers, deviceNumbers,
                slotNumbers, serialNumbers, visaRsrcs, &numArbsFound) == VI_SUCCESS)
            {
                for(ViInt16 i = 0; i < numArbsFound; ++i){
                    String *s = new String(visaRsrcs[i]);
                    resources->Add(s);
                    StringBuilder *sb = new StringBuilder();
                    sb->AppendFormat(S"Agilent N603x, {0}, bus: {1}", new String(serialNumbers[i]), __box(busNumbers[i]));
                    sb->AppendFormat(S"device: {0}, slot: {1}", __box(deviceNumbers[i]), __box(slotNumbers[i]));
                    rsrcIdTable->Add(s, sb->ToString());
                }

                result = true;
            }
        }
    }

    return result;
}

/// Given a Visa resource, return the corresponding instrument description.  Before calling this 
/// function, you should have previously called EnumQarbResources.
/// \param resource The Visa resource corresponding to the instrument whose description you
/// want.
/// \return The instrument description or a null string if we can't find a description
/// corresponding to resource.
String* QarbDiscovery::GetIdForQarbAtRsrc(String *resource)
{
    String *result = 0;

    if(rsrcIdTable->Contains(resource)){
        result = static_cast<String*>(rsrcIdTable->get_Item(resource));
    }

    return result;
}

//////////////////////////////////////////////////////////////////////
// $Log: QarbDiscoveryLibrary.cpp,v $
// Revision 1.3  2005/11/30 22:08:59  doomer
// formatting qarb discovery description to include all the model
// numbers
//
// Revision 1.2  2005/11/29 17:49:29  doomer
// documentation
//
// Modified Files:
// 	QarbDiscovery/QarbDiscovery.cpp
// 	QarbDiscovery/QarbDiscoveryHelper/QarbDiscoveryHelper.cpp
// 	QarbDiscovery/QarbDiscoveryHelper/QarbDiscoveryHelper.h
// 	QarbDiscovery/QarbDiscoveryLibrary/QarbDiscoveryLibrary.cpp
// 	QarbDiscovery/QarbDiscoveryLibrary/QarbDiscoveryLibrary.h
//
//////////////////////////////////////////////////////////////////////
