/*===========================================================================
  Module:		
     IQView.h

  Description:	
     This header file defines the types and definitions used to interface to 
     the IQView tester.

  ===========================================================================*/

#ifndef IQVIEW_H
#define IQVIEW_H

#include "IQView_defines.h"

#ifdef IQV_DLL_EXPORTS
 #define IQV_DLL_API extern __declspec(dllexport)
#else
 #ifdef IQV_C
  // Set this compile switch (IQV_C) if running C (using plain export names).
  #define IQV_DLL_API extern
 #else
  // Do not set this compile switch if running C++ (using decorated export 
  // names).
  #define IQV_DLL_API extern "C" __declspec(dllimport) 
 #endif
#endif

/*===========================================================================
  Definitions
  ===========================================================================*/

#define IQV_BOOL8       bool
#define IQV_CHAR8       char
#define IQV_UCHAR8      unsigned char
#define IQV_INT32       int
#define IQV_UINT32      unsigned int
#define IQV_FLOAT32     float
#define IQV_DOUBLE64    double


/* Length of the IP address. */
#define IQV_IP_ADDR_LEN          15                      

/* General text buffer length. */
#define IQV_TXT_LEN              80

/* Maximum number of samples in channel data buffer. */ 
#define IQV_CH_DATA_LEN     (1<<20)                  

/* Defines the number of real and imag elements returned in the eye vector */
#define IQV_N_EYE_ELEMENTS          3306           

/*===========================================================================
  Enumerations                                                              
  ===========================================================================*/

/* Enumeration for PASS/FAIL values. */
enum IQV_PF_ENUM
{
  IQV_PF_FAIL,                              /* Measurement failed. */
  IQV_PF_PASS,                              /* Measurement passed. */ 
  IQV_PF_UNDEFINED                          /* Pass/fail status is undefined. */
};

/* Enumeration for ADC input switch setting. */
enum IQV_INPUT_ADC_ENUM
{
  IQV_INPUT_ADC_RF,                         /* Data captured from RF. */
  IQV_INPUT_ADC_BB,                         /* Data captured from base band. */ 
  IQV_INPUT_ADC_UNDEFINED                   /* Undefined default value. */
};

/* Enumeration for modulation switch setting. */
enum IQV_INPUT_MOD_ENUM
{
  IQV_INPUT_MOD_DIGITAL,                    /* RF TX modulation from 
                                               internal DAC. */
  IQV_INPUT_MOD_ANALOG,                     /* RF TX modulation from 
                                               external analog input. */
  IQV_INPUT_MOD_DAC_RF                      /* RF TX modulation on RF. */
      = IQV_INPUT_MOD_DIGITAL,
  IQV_INPUT_MOD_DAC_BB                      /* RF TX modulation on BB. */
      = IQV_INPUT_MOD_ANALOG,
  IQV_INPUT_MOD_UNDEFINED                   /* Undefined default value. */
};

/* Enumeration for data capture handling. */
enum IQV_DC_ENUM
{
  IQV_DC_ARM_NEW,                           /* Arm the IQview tester for new data 
                                               capture. */
  IQV_DC_ARM_RESERVED,                      /* Reserved mode. */ 
  IQV_DC_UNDEFINED                          /* Undefined default value. */
};

/* Enumeration for continuous data capture handling. */
enum IQV_DC_CONT_ENUM
{
  IQV_DC_CONT_OFF,                          /* Use when clear the continuous mode. */
  IQV_DC_CONT_ON,                           /* Use when running in continuous mode. */ 
  IQV_DC_CONT_UNDEFINED                     /* Undefined default value. */
};

/* Enumeration for signal modulation source. */
enum IQV_SOURCE_ENUM 
{
  IQV_SOURCE_WAVE,                          /* Internal modulation from 
                                               wave. */
  IQV_SOURCE_SIGNAL_GENERATOR,              /* Internal modulation from 
                                               signal generator. */
  IQV_SOURCE_UNDEFINED                      /* Undefined default modulation 
                                               source. */
};

/* Enumeration for trigger selection - 802.11 a/b/g setting. */
enum IQV_TRIG_TYPE_ENUM
{
  IQV_TRIG_TYPE_FREE_RUN,                   /* Free running ADC sampling. */
  IQV_TRIG_TYPE_EXT,                        /* ADC External Trigger selected. */
  IQV_TRIG_TYPE_IF,                         /* ADC IF Trigger selected 
                                               - trigger calibration will be
                                                 performed. */
  IQV_TRIG_TYPE_IF_NO_CAL,                  /* ADC IF Trigger selected
                                               - trigger calibration will not
                                                 be performed. */
  IQV_TRIG_TYPE_IF2,                        /* New (potentially faster) ADC IF Trigger 
											   selected - trigger calibration will be
                                                 performed. */
  IQV_TRIG_TYPE_IF2_NO_CAL                  /* New (potentially faster) ADC IF Trigger selected
                                               - trigger calibration will not
                                                 be performed. */
};

/* Enumeration for phase correction mode - 802.11 a/g setting. */
enum IQV_PH_CORR_ENUM
{
  IQV_PH_CORR_OFF        = 1,               /* Phase correction off. */
  IQV_PH_CORR_SYM_BY_SYM = 2,               /* Symbol-by-symbol correction. */
  IQV_PH_CORR_MOVING_AVG = 3                /* Moving avg. correction 
                                               - 10 symbols
                                               - Default value. */
};

/* Enumeration for channel estimate - 802.11 a/g setting. */
enum IQV_CH_EST_ENUM
{
  IQV_CH_EST_RAW       = 1,                 /* Raw Channel Estimate 
                                               - long train. */
  IQV_CH_EST_RAW_LONG  = IQV_CH_EST_RAW,    /* Same as IQV_CH_EST_RAW. */
  IQV_CH_EST_2ND_ORDER = 2,                 /* 2nd Order Polyfit. */
  IQV_CH_EST_RAW_FULL  = 3                  /* Raw Channel Estimate 
                                               - full packet
                                               - Default value. */
};

/* Enumeration for Symbol Timing Correction - 802.11 a/g setting. */
enum IQV_SYM_TIM_ENUM
{
  IQV_SYM_TIM_OFF = 1,                      /* Symbol Timing Correction Off. */
  IQV_SYM_TIM_ON  = 2                       /* Symbol Timing Correction On
                                               - Default value. */
};

/* Enumeration for Frequency Sync. Mode - 802.11 a/g setting. */
enum IQV_FREQ_SYNC_ENUM
{
  IQV_FREQ_SYNC_SHORT_TRAIN = 1,            /* Short Training Symbol. */
  IQV_FREQ_SYNC_LONG_TRAIN  = 2,            /* Long Training Symbol 
                                               - Default value. */
  IQV_FREQ_SYNC_FULL_PACKET = 3             /* Full Data Packet. */
};

/* Enumeration for Amplitude Tracking - 802.11 a/g setting. */
enum IQV_AMPL_TRACK_ENUM
{
  IQV_AMPL_TRACK_OFF = 1,                   /* Amplitude tracking off. */
  IQV_AMPL_TRACK_ON  = 2                    /* Amplitude tracking on 
                                               - Default value. */
};

/* Enumeration for Number of equalizer taps - 802.11 b setting. */
enum IQV_EQ_ENUM
{
  IQV_EQ_OFF    = 1,                        /* Equalizer Off. */
  IQV_EQ_5_TAPS = 5,                        /* 5 taps equalizer 
                                               - Default value. */
  IQV_EQ_7_TAPS = 7,                        /* 7 taps equalizer. */
  IQV_EQ_9_TAPS = 9                         /* 9 taps equalizer. */
};

/* Enumeration for DC removal - 802.11 b setting. */
enum IQV_DC_REMOVAL_ENUM
{
  IQV_DC_REMOVAL_OFF = 0,                   /* DC removal Off 
                                               - Default value. */
  IQV_DC_REMOVAL_ON  = 1                    /* DC removal On. */
};

/* Enumeration for 802.11 b method - 802.11 b setting. */
enum IQV_11B_METHOD_ENUM
{
  IQV_11B_STANDARD_TX_ACC  = 1,             /* Use 11b standard TX accuracy
                                               - Default value. */
  IQV_11B_RMS_ERROR_VECTOR = 2              /* Use 11b RMS error vector. */
};

/* Enumeration for enabling and disabling the RF. */
enum IQV_RF_ENABLE_ENUM
{
  IQV_RF_DISABLED,                          /* RF has been disabled. */
  IQV_RF_ENABLED                            /* RF has been enabled.  */
};

/* Enumeration for RF N-connector port selection. */
enum IQV_PORT_ENUM
{
  IQV_PORT_OFF   = 1,                       /* Port has been disabled. */
  IQV_PORT_LEFT  = 2,                       /* RF uses left port.      */
  IQV_PORT_RIGHT = 3                        /* RF uses right port.     */
};

/* Enumeration for enabling and disabling swapping of I and Q signals. */
enum IQV_IQ_SWAP_ENUM
{
  IQV_IQ_SWAP_DISABLED,                          /* IQ swap has been disabled. */
  IQV_IQ_SWAP_ENABLED                            /* IQ swap has been enabled.  */
};

/* Enumeration for structure IQV_MISC_TYPE. */
enum IQV_MISC_ENUM
{
  IQV_MISC_INT,                             /* Integer parameter. */
  IQV_MISC_REAL,                            /* Real parameter. */
  IQV_MISC_COMPLEX,                         /* Complex parameter. */
  IQV_MISC_TXT,                             /* Text String parameter. */
  IQV_MISC_COMPLEX_ARRAY,                   /* Real/Complex Array parameter. */
  IQV_MISC_UNUSED                           /* Unused value. */
};

/* Enumeration for setting VSA Type */
enum IQV_VSA_TYPE_ENUM
{
  IQV_VSA_TYPE_0,                           /* VSA Type 0 - default */
  IQV_VSA_TYPE_1,                           /* VSA Type 1 - recommended */
}; // See API documentation "Setting the VSA Type"

/* Enumeration for Token Handling */
enum IQV_TOKEN_ENUM
{
  IQV_TOKEN_NO_1,
  IQV_TOKEN_NO_2,
  IQV_TOKEN_CNT
};

/* Enumeration for Video AV Method, input to IQV_GROUP_FFT */
enum IQV_VIDEO_AV_METHOD_ENUM
{
  IQV_AV_PEAK_HOLD,
  IQV_AV_MEAN
};

/* Enumeration for Window Type, input to IQV_GROUP_FFT */
enum IQV_WINDOW_TYPE_ENUM
{
  IQV_WIN_BLACKMANHARRIS,
  IQV_WIN_HANNING,
  IQV_WIN_RECT,
  IQV_WIN_FLATTOP  
};

/* Enumeration for OFDM Mode */
enum IQV_OFDM_MODE_ENUM
{
  IQV_OFDM_80211_AG,
  IQV_OFDM_80211_AG_TURBO,
  IQV_OFDM_ASTM_DSRC,
  IQV_OFDM_QUARTER_RATE  
};

/* Enumeration for Packet Analysis Mode */
enum IQV_PACKET_ANALYSIS_MODE
{
  IQV_PACKET_ANALYSIS_80211_AG = 1,
  IQV_PACKET_ANALYSIS_80211_B  
};

/* Enumeration for Modulation Control mode*/
enum IQV_MODULATION_CONTROL_ENUM
{
  IQV_WAVE_DL_MOD_DISABLE,  
  IQV_WAVE_DL_MOD_ENABLE
};

/* Enumeration holding the result of function calls. */
enum IQV_ERR_ENUM
{
  IQV_ERR_OK,                               /* The command was executed 
                                               successfully. */
  IQV_ERR_GENERAL_ERROR,                    /* The command was not executed 
                                               successfully. */
  IQV_ERR_INVALID_HNDL,                     /* The handle supplied was not 
                                               recognized as being a valid 
                                               handle. */
  IQV_ERR_UNKNOWN_PARM,                     /* The parameter supplied was 
                                               unknown. */
  IQV_ERR_ALREADY_CONNECTED,                /* A connection could not be set 
                                               up because all connection 
                                               resources are already used 
                                               up. */
  IQV_ERR_NO_ROUTE_TO_TESTER,               /* No connection to the tester 
                                               could be established. */
  IQV_ERR_OUT_OF_MEM,                       /* The required memory could not 
                                               be allocated. */
  IQV_ERR_TYPE_MISMATCH,                    /* The user requested a data type, 
                                               which could not be retrieved, 
                                               e.g., the user requested the 
                                               value of a result parameter 
                                               containing a text string, but 
                                               did not supply a buffer for it. 
                                               As another example, this error 
                                               could mean that the user tried 
                                               to retrieve an integer 
                                               parameter as a text string. 
                                               If possible a type cast will 
                                               be performed. */
  IQV_ERR_INVALID_SCENARIO,                 /* The scenario ID chosen does not 
                                               exist. Create a new scenario, 
                                               for instance by using the 
                                               command IQV_set_status. */
  IQV_ERR_DATA_CAPTURE,                     /* No data was captured during a 
                                               data capture, e.g. due to 
                                               enabling the video trigger 
                                               without supplying a signal. */
  IQV_ERR_EMPTY_VECTOR,						/* Data was requested (using IQV_result_data_real 
											   or IQV_result_data_imag), which existed, however 
											   the length of the returned vector array 
											   was zero (empty vector). */
  IQV_ERR_TOKEN,						    /* The token was invalid, general token error. */
  IQV_ERR_TIMEOUT,                          /* General timeout error. */
  IQV_ERR_N_ERRORS                          /* The number of error messages. */
};

/*===========================================================================
  Type Definitions for structures and unions holding numbers
  ===========================================================================*/

/* Structure defining a complex number. */
typedef struct
{
  double real;                              /* The real part of a complex 
                                               number. */
  double imag;                              /* The imaginary part of a complex 
                                               number. */
} IQV_COMP_NO_TYPE;

/* Structure defining a real number. */
typedef struct
{
  double real;                              /* A real number. */
} IQV_REAL_NO_TYPE;

/* Structure defining parameter used in function IQV_misc_multi_cmd_1. */
typedef struct
{
  void         *ptr_real;                   /* Pointer to real data. */
  void         *ptr_imag;                   /* Pointer to imag data. */
  unsigned int  len;                        /* Number of elements that ptr_real
                                               (and ptr_imag) points to. */
  unsigned int  type;                       /* Type of data (see 
                                               IQV_MISC_ENUM). */
} IQV_MISC_TYPE;

/* Union holding a complex or a real data value. */
typedef union
{
  IQV_COMP_NO_TYPE cval;
  IQV_REAL_NO_TYPE rval;
} IQV_COMP_TYPE;

/* Union holding a complex or a real data value. */
typedef struct
{
  IQV_COMP_TYPE    val;                     /* Value. */
  IQV_REAL_NO_TYPE lmin_reserved;           /* Minimum limit. - reserved value */
  IQV_REAL_NO_TYPE lmax_reserved;           /* Maximum limit. - reserved value */
  int              pass_reserved;           /* lmin <= val <= lmax gives true.
                                               - reserved */
  char             unit[IQV_TXT_LEN+1];     /* Text buffer for Unit. */
  char             desc[IQV_TXT_LEN+1];     /* Text buffer for Description. */
  double           reserved_00;             /* Reserved value. */
  double           reserved_01;             /* Reserved value. */
  double           reserved_02;             /* Reserved value. */
  double           reserved_03;             /* Reserved value. */
  double           reserved_04;             /* Reserved value. */
  double           reserved_05;             /* Reserved value. */
  double           reserved_06;             /* Reserved value. */
  double           reserved_07;             /* Reserved value. */
  double           reserved_08;             /* Reserved value. */
  double           reserved_09;             /* Reserved value. */                  
} IQV_RESULT_TYPE;

/*===========================================================================
  Type Definitions for structures holding measurement setups.
  ===========================================================================*/

/* Structure for RX setup parameters. */
typedef struct
{
  double freq;                              /* RF RX/TX frequency in Hz. */
  double gain;                              /* General RF RX/TX gain in dB. */
  int    rf_enabled;                        /* Use values from enumeration
                                               IQV_RF_ENABLE_ENUM for 
                                               enabling or disabling RF. */
  int    rf_port;                           /* Set RF port selection for
                                               n-connector: see IQV_PORT_ENUM. */
  double ampl;                              /* RF RX signal amplitude in dBm
                                               - not valid for TX. */
  int vsa_type;                             /* VSA Type        */
  int reserved_02;                          /* Reserved value. */
  double reserved_03;                       /* Reserved value. */
  double reserved_04;                       /* Reserved value. */
  double reserved_05;                       /* Reserved value. */
  double reserved_06;                       /* Reserved value. */
  double reserved_07;                       /* Reserved value. */
  double reserved_08;                       /* Reserved value. */
  double reserved_09;                       /* Reserved value. */                  
} IQV_RX_TX_TYPE;

/* Structure for ADC related parameters. */
typedef struct
{
  double gain_base_band;                    /* Value for base band gain in dB. 
                                               Valid when capturing data from 
                                               base band (VGA gain). */
  double sampling_time;                     /* ADC Sampling time in seconds. */
  double sampling_freq;                     /* ADC Sampling frequency in Hz.
                                               - use 66 MHz. */
  int    sampling_trig_type;                /* ADC Sampling trigger type 
                                               (See IQV_TRIG_TYPE_ENUM). */
  double sampling_trig_pre_time;            /* ADC Trigger Pre Time sets the amount
                                               of data to show before the trigger
                                               point in seconds. */
  double sampling_trig_timeout;             /* Trigger timeout in secs. */
  double sampling_trig_level;               /* Trigger level in dB relative 
                                               to VSA amplitude */
  double reserved_03;                       /* Reserved value. */
  double reserved_04;                       /* Reserved value. */
  double reserved_05;                       /* Reserved value. */
  double reserved_06;                       /* Reserved value. */
  double reserved_07;                       /* Reserved value. */
  double reserved_08;                       /* Reserved value. */
  double reserved_09;                       /* Reserved value. */                  
} IQV_ADC_TYPE;

/* Structure for DAC related parameters. */
typedef struct
{
  double           gain;                    /* Signal modulation gain in dB. */
  double           sg_freq;                 /* Internal signal generator 
                                               frequency in Hz. */
  double           sampling_freq;           /* DAC Sampling frequency in Hz.
                                               - use 80 MHz. */
  int              source;                  /* Signal source. Use value from 
                                               IQV_SOURCE_ENUM. */
  int              wave_preset;             /* If supplying a wave as source, 
                                               set to a value between 100 and 
                                               199 (both included). This 
                                               preset number is used to 
                                               select between up to 100 user 
                                               defined waves. */
  double           wave_preset_interval;    /* If supplying a wave as source,
                                               it is possible to extend the
                                               default interval between the 
                                               frames by setting this number. 
                                               The number is in sec. Values 
                                               between 0 and 0.003 are allowed. */
  double           sg_dc_error_I;           /* Signal modulation DC offset in 
                                               volt for I channel. */
  double           sg_dc_error_Q;           /* Signal modulation DC offset in 
                                               volt for Q channel. */
  double           sg_gain_error;           /* Signal Generator gain error. */
  double           sg_phase_error;          /* Signal Generator phase error. */
  double           sg_delay_error;          /* Signal Generator delay error. */
  double           reserved_02;             /* Reserved value. */
  double           reserved_03;             /* Reserved value. */
  double           reserved_04;             /* Reserved value. */
  double           reserved_05;             /* Reserved value. */
  double           reserved_06;             /* Reserved value. */
  double           reserved_07;             /* Reserved value. */
  double           reserved_08;             /* Reserved value. */
  double           reserved_09;             /* Reserved value. */                  
} IQV_DAC_TYPE;

/* Structure for FFT analysis related parameters. */
typedef struct
{
  double           res_bw;                  /* Resolution bandwidth in Hz. */
  double           video_bw;                /* Video bandwidth in Hz. */
  int              video_av_method;         /* See IQV_VIDEO_AV_METHOD_ENUM. */
  int              window_type;             /* See IQV_WINDOW_TYPE_ENUM. */
  double           f_sample;                /* Sample frequency in Hz. */
  double           nfft;                    /* Number of points in FFT. */
  double           nfft_over;               /* Minimum oversampling factor. */
  double           freq_start;              /* Start frequency in Hz */
  double           freq_stop;               /* Stop frequency in Hz */
  double           freq_delta;              /* Frequency increment in Hz */
  double           reserved_01;             /* Reserved value. */
  double           reserved_02;             /* Reserved value. */
  double           reserved_03;             /* Reserved value. */
  double           reserved_04;             /* Reserved value. */
  double           reserved_05;             /* Reserved value. */
  double           reserved_06;             /* Reserved value. */
  double           reserved_07;             /* Reserved value. */
  double           reserved_08;             /* Reserved value. */
  double           reserved_09;             /* Reserved value. */                  
} IQV_FFT_STRUCT_TYPE;



/*===========================================================================
  Main Type Definitions
  ===========================================================================*/

/* Structure used for storing an IP address. */
typedef struct
{
  char         addr [IQV_IP_ADDR_LEN+1];    /* ASCII string containing the IP 
                                               address, 
                                               e.g. xxx.xxx.xxx.xxx. */
  unsigned int port;                        /* Reserved for future use. 
                                               Make sure that it is set to 
                                               0. */
} IQV_IP_ADDR_TYPE;

/* Structure for general handle to user parameters. */
typedef struct
{
  IQV_RX_TX_TYPE rx;                        /* RX setup parameters. */
  IQV_RX_TX_TYPE tx;                        /* TX setup parameters. */
  IQV_ADC_TYPE   adc;                       /* Parameters for setup of ADC 
                                               data acquisition. */
  IQV_DAC_TYPE   dac;                       /* Parameters for setup of DAC 
                                               signal generation. */
  int            select_input_mod;          /* Use values from 
                                               IQV_INPUT_MOD_ENUM. */
  int            select_input_adc;          /* Use values from 
                                               IQV_INPUT_ADC_ENUM. */ 
  int            ph_corr_mode;              /* 802.11 a/g Setting - 
                                               Phase Correction Mode. 
                                               Use values from 
                                               IQV_PH_CORR_ENUM. */
  int            ch_estimate;               /* 802.11 a/g Setting - 
                                               Channel Estimate. 
                                               Use values from
                                               IQV_CH_EST_ENUM. */
  int            sym_tim_corr;              /* 802.11 a/g Setting - 
                                               Symbol Timing Correction. 
                                               Use values from
                                               IQV_SYM_TIM_ENUM. */
  int            freq_sync;                 /* 802.11 a/g Setting - 
                                               Frequency Sync. Mode. 
                                               Use values from
                                               IQV_FREQ_SYNC_ENUM. */
  int            eq_taps;                   /* 802.11 b Setting - 
                                               Number of equalizer taps.
                                               Use values from 
                                               IQV_EQ_ENUM. */
  int            dc_removal;                /* 802.11 b Setting - 
                                               Remove DC before analysing
                                               EVM. 
                                               Use values from 
                                               IQV_DC_REMOVAL_ENUM. */
  int            ampl_track;                /* 802.11 a/g setting - 
                                               Amplitude tracking. 
                                               Use values from 
                                               IQV_AMPL_TRACK_ENUM. */
  int            method_11b;                /* 802.11 b setting - 
                                               EVM method for 11 b.
                                               Use values from 
                                               IQV_11B_METHOD_ENUM. */
  int            iq_swap;                   /* IQ Swap setting 
                                               See IQV_IQ_SWAP_ENUM. */
  int            OFDM_demod_on;             /* OFDM Demodulation, 1:on, 0:off */
  int            OFDM_mode;                 /* OFDM Mode. Use settings 
                                               from IQV_OFDM_MODE_ENUM */   
  int            modulation_enable;         /* VSG wave load modulation mode. 
                                               See documentation on 
                                               "RF Modulation Control" */  
  double         reserved_06;               /* Reserved value. */
  double         reserved_07;               /* Reserved value. */
  double         reserved_08;               /* Reserved value. */
  double         reserved_09;               /* Reserved value. */                  
} IQV_HNDL_TYPE;

/*===========================================================================
  Variables
  ===========================================================================*/

/*===========================================================================
  Prototypes of Initializing and Terminating functions
  ===========================================================================*/
  
/*---------------------------------------------------------------------------*/
IQV_DLL_API void IQV_init (void);
/*===========================================================================
   Description:
      Use this function to initialize data structures internal to the IQview 
      Programming Interface DLL.

      Note: This function must be called before using any other IQview 
      function described in section 3, since this function allocates internal 
      data structures necessary for running the software.   
            
   Input:
      None
   
   Output:
      None
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API void IQV_term (void);
/*===========================================================================
   Description:
      Use this function to terminate the DLL. This will clean up 
      initialized/allocated data structures internally to the IQview 
      Programming Interface DLL.

      Note: This function must be called after use of the DLL.
  
   Input:
      None   
   
   Output:
      None   
  
  ===========================================================================*/

/*===========================================================================
  Connection handling functions
  ===========================================================================*/
  
/*---------------------------------------------------------------------------*/
IQV_DLL_API IQV_HNDL_TYPE *IQV_con_open (IQV_IP_ADDR_TYPE *ip_addr);
/*===========================================================================
   Description:
      Use this function to request the setup of a connection to the tester 
      with the IP address given in ip_addr. Calibration data will not 
      be loaded. Use the function IQV_con_cal_update to load the calibration
      data.

      If successful, the function will return a handle to a structure of type 
      IQV_HNDL_TYPE containing all the setup data for this tester session. 
      The setup data will be preset to factory conditions. The error level 
      will be set to IQV_ERR_OK. The user is responsible for loading the
      calibration files by calling function IQV_con_cal_update.

      A value of NULL will be returned if no connection to the tester/session 
      could be set up. In this case, the error level is set to 
      IQV_ERR_NO_ROUTE_TO_TESTER.

      If it was not possible to set up required data structures, then the 
      error level is set to IQV_ERR_OUT_OF_MEM.

      If no more handles could be allocated, the error level is set to 
      IQV_ERR_ALREADY_CONNECTED..

      If a session is already established to a tester with the given IP 
      address, the IQview Programming Interface DLL will attempt to reuse the 
      already established session. If this is successfully done, the IQview 
      Programming Interface DLL will return the same handle.

      The user is responsible for closing the session/connection by calling 
      IQV_con_close and thereby cleaning up internal data structures 
      associated with the session/connection when no longer needed.
            
   Input:
      ip_addr  needs to contain the IP address of the tester, to which a 
                connection is wanted.
   
   Output:
      Handle   the allocated connection/session handle or NULL if no 
                connection could be set up.

      Typical error levels (can be read after calling this function using the 
      function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_NO_ROUTE_TO_TESTER
      - IQV_ERR_OUT_OF_MEM
      - IQV_ERR_ALREADY_CONNECTED
  
  ===========================================================================*/
  
/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_con_close (IQV_HNDL_TYPE *hndl);
/*===========================================================================
   Description:
      Use this function to close an already established connection. Internal 
      data structures allocated by IQV_con_open for internal handling of the 
      connection will be deallocated.

      The value of the input handle will not be changed, and it is the 
      responsibility of the calling function to set it to NULL after 
      execution of IQV_con_close if so desired.

      If an error should occur, e.g. if the handle is found to be invalid, 
      the value IQV_ERR_INVALID_HNDL will be returned. If no error occurred, 
      the value IQV_ERR_OK will be returned.
            
   Input:
      handle  handle to tester session
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_con_cal_update (IQV_HNDL_TYPE *hndl);
/*===========================================================================
   Description:
      Use this function to update the calibration data from the tester. This
      function should always be called at least once after opening the 
      connection to the tester the first time. It is not necessary to call
      this function every time a connection is opened to a tester, unless 
      calibration data from another tester has been loaded prior.
      
      NOTE: Use of this function should always be followed by a call of 
            the function IQV_set_status.

      If an error should occur, e.g. if the handle is found to be invalid, 
      the value IQV_ERR_INVALID_HNDL will be returned. If no error occurred, 
      the value IQV_ERR_OK will be returned.
            
   Input:
      handle    handle to tester session
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
  
  ===========================================================================*/

/*===========================================================================
  Test setup handling functions
  ===========================================================================*/
  
/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_set_default (IQV_HNDL_TYPE *hndl);
/*===========================================================================
   Description:
     This function is used to reset the tester setup to default values. The 
     function needs a valid handle in order to be able to reset.

     All allocated scenarios will be reset. However, scenario 0 will be 
     created.

     If an error should occur, e.g. if the handle is found to be invalid, the 
     value IQV_ERR_INVALID_HNDL will be returned. If no error occurred, the 
     value IQV_ERR_OK will be returned.

     Note: The function IQV_set_default should be called at least once prior 
     to calling the function IQV_set_status. Calling IQV_set_default will 
     reset any parameter changed by the Test Suite. It is advised to call 
     this function immediately after program initialization.
            
   Input:
      Handle  handle to tester session
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_set_status (IQV_HNDL_TYPE *hndl, 
                                unsigned int   scenario);
/*===========================================================================
   Description:
      When having opened a session/connection using IQV_con_open, the user 
      has a handle to the connection. This handle points to the data, which 
      are used to set the different parameters of the tester.

      The user can utilize the handle to read the current settings, and to 
      change any of the possible parameters by writing to the data structure 
      pointed to by the handle. The user can get status/setting data from 
      another scenario by calling the function IQV_get_status.

      Having changed parameters in the data structure, the user can request 
      the tester to execute the settings in order to reflect the changes. 
      Calling the function IQV_set_status, using the handle hndl, does this.  
      The scenario input parameter for the function is used to tell the 
      tester, that these settings belong to a specific scenario.

      Parameters changed by the Test Suite may be changed by the tester 
      after executing the changes. This may for instance occur, if an illegal 
      value is written to a parameter. This does not constitute an error as 
      such, and the tester will change it to a legal value, which is being 
      the best compromise.

      If an error should occur, e.g. if the handle is found to be invalid, the 
      value IQV_ERR_INVALID_HNDL will be returned. If no error occurred, the 
      value IQV_ERR_OK will be returned.

      Calling this function will reset captured data and analyzed results for 
      the chosen scenario. This means that the user should always capture data 
      and perform data analysis after calling this function.

      Note: It is advised that the function IQV_set_default is called at least 
      once prior to calling the function IQV_set_status.
            
   Input:
      Handle    handle to tester session
      Scenario  the ID of the scenario to work with
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_OUT_OF_MEM
      - IQV_ERR_INVALID_HNDL
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_get_status (IQV_HNDL_TYPE *hndl, 
                                unsigned int   scenario);
/*===========================================================================
   Description:
      This function is used to retrieve the settings that were set by the 
      function IQV_set_status. For instance, if the user has used a scenario, 
      e.g. 3, and later has set up the tester differently with a new scenario, 
      e.g. 2, then it is possible to call the function IQV_get_status to 
      retrieve the tester settings related to scenario 3. Retrieving the data 
      means that the data will be accessible by using the handle, hndl, and 
      the settings from scenario 2 will be overwritten by the settings from 
      scenario 3. The settings are only retrieved for reading and not 
      executed. The function IQV_set_status will have to be called in order 
      to execute the retrieved settings.
      
      NOTE: It is always a good idea to call this function before any other 
      function when switching between two scenarios. This for instance means 
      that if the user has been working on scenario 1 data, then it is advised 
      to call the function when returning to work on a previous scenario, 
      e.g. 0. This is because some functions will update the stored scenario 
      status data. For instance, a call to function IQV_data_analyze will 
      update stored data for a scenario.

      If the scenario was found to be invalid, the settings will be left as 
      is. The error level IQV_ERR_INVALID_SCENARIO will be returned. 

      If an error should occur, e.g. if the handle is found to be invalid, 
      the value IQV_ERR_INVALID_HNDL will be returned. If no error occurred, 
      the value IQV_ERR_OK will be returned.
            
   Input:
      Handle    handle to tester session
      Scenario  the ID of the scenario to work with
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_INVALID_HNDL
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_get_tester_status (IQV_HNDL_TYPE *hndl, 
                                        unsigned int   scenario);
/*===========================================================================
   Description:
      This function is used to retrieve the current settings of the tester. 
      For instance, if another application should make changes to the tester
      configuration, these changes can be retrieved using IQV_get_tester_status. 

      After calling IQV_get_tester_status with a scenario, the current tester 
      configuration will be related to that scenario and the user can then use 
      the function IQV_get_status with the same scenario to retrieve the settings. 
      
      If an error should occur, e.g. if the handle is found to be invalid, 
      the value IQV_ERR_INVALID_HNDL will be returned. If no error occurred, 
      the value IQV_ERR_OK will be returned.
            
   Input:
      Handle    handle to tester session
      Scenario  the ID of the scenario to work with
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_INVALID_HNDL
  
  ===========================================================================*/



/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_set_auto_range (IQV_HNDL_TYPE *hndl, 
                                    unsigned int   scenario);
/*===========================================================================
   Description:
      Use this function to execute the settings similarly to the way 
      IQV_set_status does. In addition, the function automatically sets the 
      gain levels for the connection given by the handle.

      If an error should occur, e.g. if the handle is invalid, the value 
      IQV_ERR_INVALID_HNDL will be returned. Alternatively, if the scenario 
      is not recognized, then the error IQV_ERR_INVALID_SCENARIO will be 
      returned. If no error occurred, the value IQV_ERR_OK will be returned.

      Calling this function will reset captured data and analyzed results for 
      the chosen scenario. This means that the user should always capture 
      data and perform data analysis after calling this function.

      Note: It is advised that the function IQV_set_default is called at least 
      once prior to calling the function IQV_set_status.
            
   Input:
      Handle    handle to tester session
      Scenario  the ID of the scenario to work with
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/                                  
IQV_DLL_API int IQV_set_frame_cnt (IQV_HNDL_TYPE *hndl, 
                                   unsigned int frame_cnt);
/*===========================================================================
   Description:
      Use this function to set the number of frames (1 to 65534) that will be
      transmitted. The wanted number of frames will be transmitted 
      immediately. 

      See the documentation on "RF Modulation Control". 

      If an error should occur, e.g. if the handle is invalid, the value 
      IQV_ERR_INVALID_HNDL will be returned. If no error occurred, the value 
      IQV_ERR_OK will be returned.

   Input:      
      hndl       handle to tester session
      frame_cnt  the new number of frames to send in the next transmission 
                  (1-65534)  set to 0 to run infinitely
                  
   Output:                  
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL

  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_set_wave (IQV_HNDL_TYPE *hndl,
                              double        *real,
                              double        *imag,
                              unsigned int   n_elements);
/*===========================================================================
   Description:
      Use this function to pack the real and imaginary parts of wave data 
      (I and Q channel data) to be transmitted from the tester. The function 
      will take buff_len elements from the addresses pointed to by buff_real 
      and buff_imag, and pack this data as complex numbers to the internal 
      wave buffer. No more than IQV_CH_DATA_LEN data elements will be packed.

      If an error should occur, e.g. if the handle is invalid, the value 
      IQV_ERR_INVALID_HNDL will be returned. In case, the pointer buff_real 
      is found to be invalid, the error IQV_ERR_GENERAL_ERROR is returned. 
      If no error occurred, the value IQV_ERR_OK will be returned.
            
   Input:
      hndl       handle to tester session
      buff_real - pointer to real data
      buff_imag - pointer to imaginary data
      buff_len  - the maximum number of data elements in each of the buffers.
   
   Output:
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_GENERAL_ERROR
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_OUT_OF_MEM
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_set_wave_preset (IQV_HNDL_TYPE *hndl,
                                     unsigned int scenario);
/*===========================================================================
   Description:
      Use this function to download a predefined wave to the tester. Set the
      variable wave_preset in IQV_DAC_TYPE to a value in IQV_SIGNAL_ENUM.
      
      Example:
      
        hndl->dac.wave_preset = wave_preset;
        IQV_set_wave_preset (hndl, scenario);

      If an error should occur, e.g. if the handle is invalid, the value 
      IQV_ERR_INVALID_HNDL will be returned. 
      If no error occurred, the value IQV_ERR_OK will be returned.
            
   Input:
      hndl      handle to tester session
      Scenario  the ID of the scenario to work with
   
   Output:
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_INVALID_HNDL
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_set_wave_name (IQV_HNDL_TYPE *hndl,
                                   unsigned int scenario,
                                   char *name);
/*===========================================================================
   Description:
      Use this function to download a *.mod file to the tester. Use the file
      name of the *.mod file as input parameter. This could be a file 
      generated by IQsignal from the File->Save Generator File menu.
      
      Example:
      
        IQV_set_wave_name (hndl, scenario, "wave1.mod");

      If an error should occur, e.g. if the handle is invalid, the value 
      IQV_ERR_INVALID_HNDL will be returned. 
      If no error occurred, the value IQV_ERR_OK will be returned.
            
   Input:
      hndl      handle to tester session
      Scenario  the ID of the scenario to work with
      name     - the file name of the *.mod file to use
   
   Output:
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_INVALID_HNDL
  
  ===========================================================================*/

/*===========================================================================
  Data Capture and Signal Analysis handling
  ===========================================================================*/
  
/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_data_capture (IQV_HNDL_TYPE *hndl, 
                                  unsigned int   scenario,
                                  unsigned int   capture_mode);
/*===========================================================================
   Description:
      Use this function to execute a capture of data from the tester. The 
      test setup used will be that set up using the function IQV_set_status. 
      The input scenario is used to tie the captured data to the scenario 
      given in IQV_set_status.
      
      If an error should occur, e.g. if the handle is invalid, the value 
      IQV_ERR_INVALID_HNDL will be returned. Alternatively, if the scenario 
      is not recognized, then the error IQV_ERR_INVALID_SCENARIO will be 
      returned. If no error occurred, i.e. data was captured, the value 
      IQV_ERR_OK will be returned. 
      
      The function will wait approx. two (2) sec. for a trigger to happen 
      before returning. Use the function IQV_result_data_valid to verify if 
      data was captured. This makes it possible to decide whether to retry 
      the data capture or to cancel it. Each time the function is 
      called, the trigger will be armed.
      
      Always call this function with capture_mode set to IQV_DC_ARM_NEW. 
      Other modes are reserved for future use. 
      
      After a successful data capture, the raw data samples can be read using 
      the result handling functions IQV_result_data_real and 
      IQV_result_data_imag.

   Input:
      hndl      handle to tester session
      scenario  the ID of the scenario to work with
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_INVALID_HNDL
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_data_capture_cont (IQV_HNDL_TYPE *hndl, 
                                       unsigned int   scenario,
                                       unsigned int   capture_mode);
/*===========================================================================
   Description:
      Use this function when executing several data captures of data from the 
      tester in a row. The test setup used will be that set up using the 
      function IQV_set_status. The input scenario is used to tie the captured 
      data to the scenario given in IQV_set_status.
      
      This function differs from IQV_data_capture in the fact that the 
      function uses double buffering of data. This way some data processing 
      on the PC may occur in parallel with the IQview tester capturing the 
      next set of data. This may help speed up the test execution.
      
      Start the double buffered data capture session by calling this function
      with capture_mode set to IQV_DC_CONT_ON. During the first call, this 
      will request fresh data from the tester and then initialize an 
      additional fetch of data that will run in the background. Subsequent 
      calls should be made with the capture_mode set to IQV_DC_CONT_ON. Each 
      call will return data from the previous fetch and initiate a new fetch 
      in the background. 
      
      NOTE: When done performing the data captures, always call this function 
      once with the capture_mode set to IQV_DC_CONT_OFF.
      
      WARNING: Do not call any function, which sets up the tester, e.g.
      IQV_set_default or IQV_set_status, before after calling the 
      IQV_data_capture_cont with the parameter IQV_DC_CONT_OFF.
      
      If an error should occur, e.g. if the handle is invalid, the value 
      IQV_ERR_INVALID_HNDL will be returned. Alternatively, if the scenario 
      is not recognized, then the error IQV_ERR_INVALID_SCENARIO will be 
      returned. If no error occurred, i.e. data was captured, the value 
      IQV_ERR_OK will be returned. 
      
      The function will wait approx. two (2) sec. for a trigger to happen 
      before returning. Use the function IQV_result_data_valid to verify if 
      data was captured. This makes it possible to decide whether to retry 
      the data capture or to cancel it. Each time the function is 
      called, the trigger will be armed.
      
      After a successful data capture, the raw data samples can be read using 
      the result handling functions IQV_result_data_real and 
      IQV_result_data_imag.

   Input:
      hndl          handle to tester session
      scenario      the ID of the scenario to work with
      capture_mode - Use values from IQV_DC_CONT_ENUM
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_INVALID_HNDL
  
  ===========================================================================*/


/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_set_arbitrary_wave (IQV_HNDL_TYPE *hndl, 
                                        unsigned int   scenario,
                                        double        *buff_real,
                                        double        *buff_imag,
                                        unsigned int   buff_len);
/*===========================================================================
   Description:
      Use this function to perform an analysis on raw wave data (e.g. without 
      sending it through the tester). After calling this function, the user 
      can perform an analysis on the wave, in the same manner as for a wave 
      captured from the tester. Note: the sampling frequency of the supplied 
      wave should be 66 MHz. 

   Input:
     hndl  handle to tester session
     scenario  the ID of the scenario to work with
     buff_real - pointer to real data
     buff_imag - pointer to imaginary data
     buff_len  - the maximum number of data elements in each of the buffers. 
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_INVALID_SCENARIO
  
 ===========================================================================*/


/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_copy_capture_selection (IQV_HNDL_TYPE *hndl, 
                                        unsigned int   scenario,
                                        unsigned int   dest_scenario,
                                        unsigned int   start_sample,
                                        unsigned int   length);
/*===========================================================================
   Description:
      Use this function to move a selected portion of the data capture 
      (and hndl settings) to another scenario. 

   Input:
    hndl 	- 	    handle to the tester
    scenario - 	    scenario ID
    dest_scenario - destination scenario ID
    start_sample - 	Sample index to start data copy from
    length -        Number of samples to copy

   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_GENERAL_ERROR
  
 ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_truncate_capture (IQV_HNDL_TYPE *hndl, 
                                        unsigned int   scenario,
					unsigned int   start_sample,
                                        unsigned int   length);
/*===========================================================================
   Description:
      Use this function to truncate the data capture.       

   Input:
    hndl 	- 	    handle to the tester
    scenario - 	    scenario ID
    start_sample - 	Sample index to start data copy from
    length -        Number of samples to copy

   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_GENERAL_ERROR
  
 ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_load_wave_as_capture (IQV_HNDL_TYPE *hndl, 
                                          unsigned int scenario,
                                          char *name);
/*===========================================================================
   Description:
      Use this function to perform an analysis on pre-existing wave data (e.g. 
      without sending it through the tester). After calling this function, the 
      user can perform an analysis on the wave, in the same manner as for a 
      wave captured from the tester. Note: the sampling frequency of the 
      supplied wave should be 66 MHz. 

   Input:
     hndl      handle to tester session
     scenario  the ID of the scenario to work with
     name     - the file name of the *.mod or *.sig file to use
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_INVALID_SCENARIO
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_load_wave_as_capture_atten (IQV_HNDL_TYPE *hndl, 
                                          	unsigned int scenario,
                                          	char *name, 
					  	double *atten);
/*===========================================================================
   Description:
      This function can be used with capture data saved using 
	  IQV_data_store_atten or IQV_data_store_atten_file_name in order to 
	  also find out the attenuation value that was saved with the capture. 
	  After calling this function, atten will point to the saved attenuation
	  value. Otherwise, this function is no different than 
	  IQV_load_wave_as_capture. If no attenuation value is available for the 
	  saved data *atten will be set to 0. 

      Use this function to perform an analysis on pre-existing wave data (e.g. 
      without sending it through the tester). After calling this function, the 
      user can perform an analysis on the wave, in the same manner as for a 
      wave captured from the tester. Note: the sampling frequency of the 
      supplied wave should be 66 MHz. 

   Input:
     hndl      handle to tester session
     scenario  the ID of the scenario to work with
     name     - the file name of the *.mod or *.sig file to use
   
   Output:
     atten	  - the attenuation value saved with the capture (otherwise, 0)
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_INVALID_SCENARIO
  
  ===========================================================================*/



/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_data_analyze (IQV_HNDL_TYPE *hndl, 
                                  unsigned int   scenario,
                                  unsigned int   test_group);
/*===========================================================================
   Description:
      Use this function to perform an analysis of the data captured in 
      function IQV_data_capture. The user supplies an analysis group number, 
      which identifies the group of analysis to perform, e.g. an analysis 
      group for measuring power or a group for analyzing EVM.

      If an error should occur, e.g. if the handle is invalid, the value 
      IQV_ERR_INVALID_HNDL will be returned. If the scenario ID is not 
      recognized, the error IQV_ERR_INVALID_SCENARIO will be returned. If the 
      analysis group is not recognized, the error IQV_ERR_UNKNOWN_PARM will be 
      returned. If the function is called without having any captured data, the 
      error IQV_ERR_DATA_CAPTURE will be returned. If no error occurred, the 
      value IQV_ERR_OK will be returned.

      After the data analysis, the data analysis results can be read using the 
      result handling functions.      
      
   Input:
      hndl            handle to tester session
      scenario        the ID of the scenario to work with
      test_group      the analysis group defining the type of analysis to 
                       perform
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_DATA_CAPTURE
      
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_data_analyze_packet (IQV_HNDL_TYPE *hndl, 
                                  unsigned int   scenario,
                                  unsigned int   dest_scenario,
                                  unsigned int   test_group,
                                  unsigned int   packet_index,
                                  unsigned int   mode);
/*===========================================================================
    Description:
      Use this function to perform an analysis of a single packet contained in 
      the data captured in function IQV_data_capture. The user supplies an 
      analysis group number, which identifies the group of analysis to perform, 
      e.g. an analysis group for measuring power or a group for analyzing EVM.

      The user must supply a mode because this function will first perform an 
      analysis (with group IQV_GROUP_802_11_B_ANALYSIS if mode is set to
      IQV_PACKET_ANALYSIS_80211_B (2), or with group IQV_GROUP_OFDM_ANALYSIS if 
      mode is set to IQV_PACKET_ANALYSIS_80211_AG (1)), in order to find the 
      start/stop pointers for each packet. Thereafter, this function will use 
      function IQV_copy_capture_selection to copy the data (and hndl settings) 
      for the selected packet into scenario number dest_scenario, and then 
      perform an analysis on data in dest_scenario using test_group. 

      The user should note that packet_index is zero (0) based. 
      
      If an error should occur, e.g. if the handle is invalid, the value 
      IQV_ERR_INVALID_HNDL will be returned. If the scenario ID is not 
      recognized, the error IQV_ERR_INVALID_SCENARIO will be returned. If the 
      analysis group is not recognized, the error IQV_ERR_UNKNOWN_PARM will be 
      returned. If the function is called without having any captured data, the 
      error IQV_ERR_DATA_CAPTURE will be returned. If the function can not 
      allocate enough memory to perform the operation, IQV_ERR_OUT_OF_MEM will
      be returned. IQV_ERR_GENERAL_ERROR will be returned in all other error 
      cases, for example, if the user requests analysis on packet six (6) and 
      there are only three (3) packets in the current data capture. 
      
      If no error occurred, the value IQV_ERR_OK will be returned.

      After the data analysis, the data analysis results can be read using the 
      result handling functions.      
      
   Input:
      hndl            handle to tester session
      scenario        the ID of the scenario to work with
      dest_scenario   the ID of the scenario to copy the packet data to
      test_group      the analysis group defining the type of analysis to 
                       perform
      packet_index   - the zero based index of the packet to analyze
      mode           - IQV_PACKET_ANALYSIS_80211_B or 
                       IQV_PACKET_ANALYSIS_80211_AG

   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_DATA_CAPTURE
      - IQV_ERR_OUT_OF_MEM
      
  ===========================================================================*/



/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_fft_analyze (IQV_HNDL_TYPE          *hndl, 
                                  unsigned int          scenario,
                                  IQV_FFT_STRUCT_TYPE   *fft_struct);
/*===========================================================================
   Description:
      Use this function to perform an FFT analysis of the data captured in 
      function IQV_data_capture. 

      If an error should occur, e.g. if the handle is invalid, the value 
      IQV_ERR_INVALID_HNDL will be returned. If the scenario ID is not 
      recognized, the error IQV_ERR_INVALID_SCENARIO will be returned. If the 
      analysis group is not recognized, the error IQV_ERR_UNKNOWN_PARM will be 
      returned. If the function is called without having any captured data, the 
      error IQV_ERR_DATA_CAPTURE will be returned. If no error occurred, the 
      value IQV_ERR_OK will be returned.

      After the data analysis, the data analysis results can be read using the 
      result handling functions.      
      
   Input:
      hndl            handle to tester session
      scenario        the ID of the scenario to work with
      fft_struct     - pointer to data structure containing FFT analysis input 
                       parameters
         
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_DATA_CAPTURE
      
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_set_default_fft_parms (IQV_FFT_STRUCT_TYPE   *fft_struct);
/*===========================================================================
   Description:
      Use this function to set default values to structure IQV_FFT_STRUCT_TYPE. 
      
   Input:
      fft_struct  pointer to structure of type IQV_FFT_STRUCT_TYPE
               
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK      
      
  ===========================================================================*/



/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_data_store (IQV_HNDL_TYPE *hndl, 
                                unsigned int   scenario);
/*===========================================================================
   Description:
      Use this function to store the captured data to "iqv.sig". This file 
      can be opened in IQsignal for analysis. Only valid data will be stored.

      If an error should occur, e.g. if the handle is invalid, the value 
      IQV_ERR_INVALID_HNDL will be returned. If the scenario ID is not 
      recognized, the error IQV_ERR_INVALID_SCENARIO will be returned. If the 
      function is called without having any valid captured data, the error 
      IQV_ERR_DATA_CAPTURE will be returned. If no error occurred, the value
      IQV_ERR_OK will be returned.
      
   Input:
      hndl            handle to tester session
      scenario        the ID of the scenario to work with
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_DATA_CAPTURE
      
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_data_store_file_name (IQV_HNDL_TYPE *hndl, 
                                          unsigned int   scenario,
                                          char          *file_name);
/*===========================================================================
   Description:
      Use this function to store the captured data to a file with the name
      given using the pointer file_name. This file can be opened in IQsignal 
      for analysis. Only valid data will be stored.

      If an error should occur, e.g. if the handle is invalid, the value 
      IQV_ERR_INVALID_HNDL will be returned. If the scenario ID is not 
      recognized, the error IQV_ERR_INVALID_SCENARIO will be returned. If the 
      function is called without having any valid captured data, the error 
      IQV_ERR_DATA_CAPTURE will be returned. If no error occurred, the value
      IQV_ERR_OK will be returned.
      
   Input:
      hndl            handle to tester session
      scenario        the ID of the scenario to work with
      file_name      - pointer to the name of the file to store the data to
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_DATA_CAPTURE
      
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_data_store_atten (IQV_HNDL_TYPE *hndl, 
					unsigned int   scenario, 
					double	ext_atten);
/*===========================================================================
   Description:
      Use this function to store the captured data to "iqv.sig", with external 
	  attenuation applied to the captured data before saving. This file can be 
	  opened in IQsignal for analysis. Only valid data will be stored.

      If an error should occur, e.g. if the handle is invalid, the value 
      IQV_ERR_INVALID_HNDL will be returned. If the scenario ID is not 
      recognized, the error IQV_ERR_INVALID_SCENARIO will be returned. If the 
      function is called without having any valid captured data, the error 
      IQV_ERR_DATA_CAPTURE will be returned. If no error occurred, the value
      IQV_ERR_OK will be returned.
      
   Input:
      hndl            handle to tester session
      scenario        the ID of the scenario to work with
	  ext_atten		 - external attenuation in dB
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_DATA_CAPTURE
      
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_data_store_atten_file_name (IQV_HNDL_TYPE *hndl, 
                                          unsigned int  scenario,
                                          char          *file_name, 
					  double	ext_atten);
/*===========================================================================
   Description:
      Use this function to store the captured data to a file with the name
      given using the pointer file_name, with external attenuation applied 
	  to the captured data before saving. This file can be opened in IQsignal 
      for analysis. Only valid data will be stored.

      If an error should occur, e.g. if the handle is invalid, the value 
      IQV_ERR_INVALID_HNDL will be returned. If the scenario ID is not 
      recognized, the error IQV_ERR_INVALID_SCENARIO will be returned. If the 
      function is called without having any valid captured data, the error 
      IQV_ERR_DATA_CAPTURE will be returned. If no error occurred, the value
      IQV_ERR_OK will be returned.
      
   Input:
      hndl            handle to tester session
      scenario        the ID of the scenario to work with
      file_name      - pointer to the name of the file to store the data to
	  ext_atten		 - external attenuation in dB
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_DATA_CAPTURE
      
  ===========================================================================*/






/*===========================================================================
  Result handling functions
  ===========================================================================*/
  
/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_result (IQV_HNDL_TYPE   *hndl, 
                            unsigned int     scenario, 
                            char            *parameter,
                            IQV_RESULT_TYPE *result);
/*===========================================================================
   Description:
      This function is used to return the values related to one particular 
      measurement. The measurement is named by the text string parameter.
 
      The function needs a pointer to an input parameter "result" of type 
      IQV_RESULT_TYPE large enough to hold the data in the type 
      IQV_RESULT_TYPE. 

      The error level will be set to IQV_ERR_INVALID_HNDL, if the handle 
      supplied by the Test Suite is not recognized. Similarly, the error level 
      will be set to IQV_ERR_INVALID_SCENARIO, if the scenario ID supplied is 
      not recognized. If the wanted parameter is not found for the scenario 
      in question, the error level IQV_ERR_UNKNOWN_PARM will be set.
      
   Input:
      hndl       handle to tester session
      scenario   the ID of the scenario to work with
      parameter - the name of the parameter to find given as an ASCII string
      result    - pointer to structure of type IQV_RESULT_TYPE
   
   Output:
      Typical error levels (can also be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_TYPE_MISMATCH
      - IQV_ERR_UNKNOWN_PARM
      - IQV_ERR_INVALID_SCENARIO
  
  ===========================================================================*/
                                                
/*---------------------------------------------------------------------------*/                                                
IQV_DLL_API double IQV_result_real (IQV_HNDL_TYPE *hndl, 
                                    unsigned int   scenario, 
                                    char          *parameter);
/*===========================================================================
   Description:
      This function is used to return the real value related to one 
      particular measurement. The measurement is named by the text string 
      parameter.

      The function will return the real value of the parameter. The error 
      level is set to IQV_ERR_OK. 

      The error level will be set to IQV_ERR_INVALID_HNDL, if the handle 
      supplied by the Test Suite is not recognized. Similarly, the error 
      level will be set to IQV_ERR_INVALID_SCENARIO, if the scenario ID 
      supplied is not recognized. If the wanted parameter is not found for 
      the scenario in question, the error level IQV_ERR_UNKNOWN_PARM will be 
      set, and the value 0 will be returned.      

   Input:
      hndl       handle to tester session
      scenario   the ID of the scenario to work with
      parameter - the name of the parameter to find given as an ASCII string
   
   Output:
      result  return value for the parameter in question
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_TYPE_MISMATCH
      - IQV_ERR_UNKNOWN_PARM
      - IQV_ERR_INVALID_SCENARIO
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/                                            
IQV_DLL_API double IQV_result_imag (IQV_HNDL_TYPE *hndl, 
                                    unsigned int   scenario, 
                                    char          *parameter);
/*===========================================================================
   Description:
      This function is used to return the imaginary value related to one 
      particular measurement. The measurement is named by the text string 
      parameter.

      The function will return the imaginary value of the parameter. The 
      error level is set to IQV_ERR_OK. 

      The error level will be set to IQV_ERR_INVALID_HNDL, if the handle 
      supplied by the Test Suite is not recognized. Similarly, the error level 
      will be set to IQV_ERR_INVALID_SCENARIO, if the scenario ID supplied is 
      not recognized. If the wanted parameter is not found for the scenario in 
      question, the error level IQV_ERR_UNKNOWN_PARM will be set, and the 
      value 0 will be returned.
      
   Input:
      hndl       handle to tester session
      scenario   the ID of the scenario to work with
      parameter - the name of the parameter to find given as an ASCII string
   
   Output:
      result  return value for the parameter in question
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_TYPE_MISMATCH
      - IQV_ERR_UNKNOWN_PARM
      - IQV_ERR_INVALID_SCENARIO
  
  ===========================================================================*/
                                            
/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_result_int (IQV_HNDL_TYPE *hndl, 
                                unsigned int   scenario, 
                                char          *parameter);
/*===========================================================================
   Description:
      This function is used to return the integer value related to one 
      particular measurement. The measurement is named by the text string 
      parameter.

      The function will return the integer value of the parameter. The error 
      level is set to IQV_ERR_OK. 

      The error level will be set to IQV_ERR_INVALID_HNDL, if the handle 
      supplied by the Test Suite is not recognized. Similarly, the error 
      level will be set to IQV_ERR_INVALID_SCENARIO, if the scenario ID 
      supplied is not recognized. If the wanted parameter is not found for the 
      scenario in question, the error level IQV_ERR_UNKNOWN_PARM will be set, 
      and the value 0 will be returned.    

   Input:
      hndl       handle to tester session
      scenario   the ID of the scenario to work with
      parameter - the name of the parameter to find given as an ASCII string
   
   Output:
      result  return value for the parameter in question
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_TYPE_MISMATCH
      - IQV_ERR_UNKNOWN_PARM
      - IQV_ERR_INVALID_SCENARIO
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_result_txt (IQV_HNDL_TYPE *hndl, 
                                unsigned int   scenario, 
                                char          *parameter,
                                char          *buff,
                                unsigned int   buff_len);
/*===========================================================================
   Description:
      This function is used to return the text string related to one 
      particular measurement. The measurement is named by the text string 
      parameter.

      The function will return the value of the parameter as a text string. 
      The text string will be extracted into the buffer supplied from the 
      Test Suite by the parameter buff. The input buff_len should be set to 
      the size of the buffer incl. room for a null terminator. This means that 
      no more than buff_len-1 characters will be copied and a null terminator 
      will be added. The error level is set to IQV_ERR_OK. 

      The error level will be set to IQV_ERR_INVALID_HNDL, if the handle 
      supplied by the Test Suite is not recognized. Similarly, the error level 
      will be set to IQV_ERR_INVALID_SCENARIO, if the scenario ID supplied is 
      not recognized. If the wanted parameter is not found for the scenario in 
      question, the error level IQV_ERR_UNKNOWN_PARM will be set, and an empty 
      string will be returned.
      
   Input:
      hndl       handle to tester session
      scenario   the ID of the scenario to work with
      parameter - the name of the parameter to find given as an ASCII string
      buff       pointer to the buffer to store the text string
      buff_len   size of the buffer including room for the null terminator
   
   Output:
      buff  text string is copied to buff
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_TYPE_MISMATCH
      - IQV_ERR_UNKNOWN_PARM
      - IQV_ERR_INVALID_SCENARIO
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API unsigned int IQV_result_data_real (IQV_HNDL_TYPE *hndl, 
                                               unsigned int   scenario, 
                                               char          *parameter,
                                               double        *buff,
                                               unsigned int   buff_len);
/*===========================================================================
   Description:
      This function is used to return the real values related to analyzed 
      data. The type of analyzed data is named by the text string parameter.

      The function can also be used to retrieve the real part of the 
      captured data samples. If this is wanted, the input parameter called 
      parameter is then set to NULL or an empty string ( or 
      IQV_RES_CAPT_DATA).

      The caller of this function is responsible for allocating a buffer 
      array of type double and for giving a pointer to the buffer in the 
      parameter buff. The variable buff_len need to specify the size of the 
      buffer in terms of elements. This means, that if there is room for 5 
      double values (i.e. 5 * 8 bytes) in the buffer, then this value needs 
      to be set to 5.

      The function will copy a maximum of buff_len double values into the 
      buffer. The actual number of elements copied will be returned from the 
      function.

      The function sets the error level IQV_ERR_OK if data is returned. The 
      error level will be set to IQV_ERR_INVALID_HNDL, if the handle supplied 
      by the Test Suite is not recognized. Similarly, the error level will be 
      set to IQV_ERR_INVALID_SCENARIO, if the scenario ID supplied is not 
      recognized. If the wanted parameter is not found for the scenario in 
      question, the error level IQV_ERR_UNKNOWN_PARM will be set, and an empty 
      buffer will be returned.
            
   Input:
      hndl       handle to tester session
      scenario   the ID of the scenario to work with
      parameter - the name of the parameter to find given as an ASCII string
      buff       pointer to the buffer to store the double values
      buff_len   size of the buffer in terms of number of double values, that 
                  can be stored in the buffer
   
   Output:
      buff        double values are copied to buff
      n_elements  number of elements copied
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_TYPE_MISMATCH
      - IQV_ERR_UNKNOWN_PARM
      - IQV_ERR_INVALID_SCENARIO
  
  ===========================================================================*/
                                                 
/*---------------------------------------------------------------------------*/                                                 
IQV_DLL_API unsigned int IQV_result_data_imag (IQV_HNDL_TYPE *hndl, 
                                               unsigned int   scenario,
                                               char          *parameter,
                                               double        *buff,
                                               unsigned int   buff_len);
/*===========================================================================
   Description:
      This function is used to return the imaginary values related to 
      analyzed data. The type of analyzed data is named by the text string 
      parameter.

      The function can also be used to retrieve the imaginary part of the 
      captured data samples. If this is wanted, the input parameter called 
      parameter is then set to NULL or an empty string ( or 
      IQV_RES_CAPT_DATA).

      The caller of this function is responsible for allocating a buffer 
      array of type double and for giving a pointer to the buffer in the 
      parameter buff. The variable buff_len need to specify the size of the 
      buffer in terms of elements. This means, that if there is room for 5 
      double values (i.e. 5 * 8 bytes) in the buffer, then this value needs 
      to be set to 5.

      The function will copy a maximum of buff_len double values into the 
      buffer. The actual number of elements copied will be returned from the 
      function.

      The function sets the error level IQV_ERR_OK if data is returned. The 
      error level will be set to IQV_ERR_INVALID_HNDL, if the handle supplied 
      by the Test Suite is not recognized. Similarly, the error level will be 
      set to IQV_ERR_INVALID_SCENARIO, if the scenario ID supplied is not 
      recognized. If the wanted parameter is not found for the scenario in 
      question, the error level IQV_ERR_UNKNOWN_PARM will be set, and an 
      empty buffer will be returned.     

   Input:
      hndl       handle to tester session
      scenario   the ID of the scenario to work with
      parameter - the name of the parameter to find given as an ASCII string
      buff       pointer to the buffer to store the double values
      buff_len   size of the buffer in terms of number of double values, 
                  that can be stored in the buffer
   
   Output:
      buff        double values are copied to buff
      n_elements  number of elements copied
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_TYPE_MISMATCH
      - IQV_ERR_UNKNOWN_PARM
      - IQV_ERR_INVALID_SCENARIO
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_result_exist (IQV_HNDL_TYPE *hndl, 
                                  unsigned int   scenario, 
                                  char          *parameter);
/*===========================================================================
   Description:
     This function is used to find out if a parameter exists for a given 
     scenario. If the parameter is found, a result different from 0 is 
     returned. Otherwise, the result 0 is returned.

     If checking whether captured data is present for the scenario, please 
     use an empty string () or NULL as parameter.

     The error level will be set to IQV_ERR_INVALID_HNDL, if the handle 
     supplied by the Test Suite is not recognized. Similarly, the error level 
     will be set to IQV_ERR_INVALID_SCENARIO, if the scenario ID supplied is 
     not recognized. Otherwise, if the parameter exists, the function sets 
     the error level IQV_ERR_OK, and if the parameter does not exist, the 
     function sets the error level to IQV_ERR_UNKNOWN_PARM.
      
   Input:
      hndl       handle to tester session
      scenario   the ID of the scenario to work with
      parameter - the name of the parameter to find given as an ASCII string
   
   Output:
      result - the parameter exists or not (0)
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_UNKNOWN_PARM
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_result_exist_valid (IQV_HNDL_TYPE *hndl, 
                                        unsigned int   scenario, 
                                        char          *parameter);
/*===========================================================================
   Description:
     This function is used to find out if a result exists for a given 
     scenario. If the parameter is found and the analysis result is valid, a 
     result different from 0 is returned. Otherwise, the result 0 is returned.

     Checking whether captured data is present for the scenario should be
     done with the function IQV_result_exist.

     The error level will be set to IQV_ERR_INVALID_HNDL, if the handle 
     supplied by the Test Suite is not recognized. Similarly, the error level 
     will be set to IQV_ERR_INVALID_SCENARIO, if the scenario ID supplied is 
     not recognized. Otherwise, if the parameter exists, the function sets 
     the error level IQV_ERR_OK, and if the parameter does not exist, the 
     function sets the error level to IQV_ERR_UNKNOWN_PARM.
      
   Input:
      hndl       handle to tester session
      scenario   the ID of the scenario to work with
      parameter - the name of the parameter to find given as an ASCII string
   
   Output:
      result - the parameter exists or not (0)
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_UNKNOWN_PARM
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_result_data_clip (IQV_HNDL_TYPE *hndl, 
                                      unsigned int   scenario);
/*===========================================================================
   Description:
     This function is used to find out if clipping occurred during data 
     capture (using IQV_data_capture).

     The function will return 1 if the clipping occurred and 0 if clipping 
     did not occur. In both cases the error level is set to IQV_ERR_OK. 

     The error level will be set to IQV_ERR_INVALID_HNDL, if the handle 
     supplied by the Test Suite is not recognized. Similarly, the error level 
     will be set to IQV_ERR_INVALID_SCENARIO, if the scenario ID supplied is 
     not recognized.

   Input:
      hndl       handle to tester session
      scenario   the ID of the scenario to work with
   
   Output:
      result - clipping (1)/no clipping (0)
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_UNKNOWN_PARM
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_result_data_valid (IQV_HNDL_TYPE *hndl, 
                                       unsigned int   scenario);
/*===========================================================================
   Description:
     This function is used to find out if data is valid after running a data 
     capture (using IQV_data_capture).

     The function will return 1 if data was captured and 0 if no data is 
     available. In both cases the error level is set to IQV_ERR_OK. 

     The error level will be set to IQV_ERR_INVALID_HNDL, if the handle 
     supplied by the Test Suite is not recognized. Similarly, the error level 
     will be set to IQV_ERR_INVALID_SCENARIO, if the scenario ID supplied is 
     not recognized.

   Input:
      hndl       handle to tester session
      scenario   the ID of the scenario to work with
   
   Output:
      result - data was captured (1)
               no data is available (no trigger event) (0)
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_INVALID_SCENARIO
      - IQV_ERR_UNKNOWN_PARM
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_result_tx_data_clipping (IQV_HNDL_TYPE *hndl);
/*===========================================================================
   Description:
     This function is used to find out if the internal TX DAC is clipping.

     The function will return 1 if TX clipping is occurring and 0 if no TX
     clipping was detected.

     In both cases, the error level is set to IQV_ERR_OK. 

     The error level will be set to IQV_ERR_INVALID_HNDL, if the handle 
     supplied by the Test Suite is not recognized.

   Input:
      hndl       handle to tester session
   
   Output:
      result    - TX DAC is clipping (1)
                - TX DAC is not clipping (0)
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_GENERAL_ERROR
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_result_tx_data_low (IQV_HNDL_TYPE *hndl);
/*===========================================================================
   Description:
     This function is used to find out if a low signal is generated on the 
     TX DAC, i.e. a few number of bits are utilized. 

     The function will return 1 if few bits are utilized and 0 otherwise.

     In both cases, the error level is set to IQV_ERR_OK. 

     The error level will be set to IQV_ERR_INVALID_HNDL, if the handle 
     supplied by the Test Suite is not recognized.

   Input:
      hndl       handle to tester session
   
   Output:
      result    - TX data is low (1)
                - Otherwise (0)
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_GENERAL_ERROR
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_result_frame_cnt_done (IQV_HNDL_TYPE *hndl);
/*===========================================================================
   Description:
     This function is used to find out if the transmitter has finished 
     transmitting the wanted number of packets after calling the 
     IQV_set_frame_cnt function. The function will return IQV_PF_FAIL if the
     transmitter is still transmitting. The function will return IQV_PF_PASS
     if the transmitter has stopped transmitting.
     
     The function can be used in a loop to wait for the conclusion of the
     transmission.
     
     The error level will be set to IQV_ERR_INVALID_HNDL, if the handle 
     supplied by the Test Suite is not recognized.      
     
   Input:
      hndl       handle to tester session
   
   Output:
      result - Done transmitting (IQV_PF_FAIL)
               Transmitting (IQV_PF_PASS)
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_result_version (IQV_HNDL_TYPE *hndl, 
                                    unsigned char *buff,
                                    unsigned int   buff_len);
/*===========================================================================
   Description:
      This function is used to get the version of different parts of the 
      system. 
     
      The function will return the value as a text string. 
      The text string will be copied into the buffer supplied from the 
      Test Suite by the parameter buff. The input buff_len should be set to 
      the size of the buffer incl. room for a null terminator. This means 
      that no more than buff_len-1 characters will be copied and a null 
      terminator will be added. The error level is set to IQV_ERR_OK. 
     
      If no connection to a tester is present, set hndl to NULL when calling
      the function. The result will be in a format similar to the following:
      
      MW: Jan 08, 2004, 1.9, BA: 1.1.0 January 13, 2004, 
      TC: 1.1.4 January 13, 2004

      If a connection to a tester is present, it is possible to supply hndl, 
      and this will result in extra information being appended. The result 
      will be in a format similar to the following:      
     
      MW: Jan 08, 2004, 1.9, BA: 1.1.0 January 13, 2004, 
      TC: 1.1.4 January 13, 2004, TST SN: IQV030325006, TST FW: v1.0.36 Rev. C
     
   Input:
      hndl       handle to tester session
      buff      - pointer to a buffer where the text string will be copied
      buff_len   size of the buffer including room for the null terminator      
   
   Output:
      buff  text string is copied to buff   
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL - if supplying a hndl      
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_result_hw_version (IQV_HNDL_TYPE *hndl, 
                                    unsigned char *buff,
                                    unsigned int   buff_len);
/*===========================================================================
   Description:
      This function is used to get the hardware version of the system. 
     
      The function will return the value as a text string. 
      The text string will be copied into the buffer supplied from the 
      Test Suite by the parameter buff. The input buff_len should be set to 
      the size of the buffer incl. room for a null terminator. This means 
      that no more than buff_len-1 characters will be copied and a null 
      terminator will be added. The error level is set to IQV_ERR_OK. 
     
      If no connection to a tester is present, set hndl to NULL when calling
      the function. The result will be in a format similar to the following:
      
      1.2.3 Config0
           
   Input:
      hndl       handle to tester session
      buff      - pointer to a buffer where the text string will be copied
      buff_len   size of the buffer including room for the null terminator      
   
   Output:
      buff  text string is copied to buff   
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL - if supplying a hndl      
  
  ===========================================================================*/


/*---------------------------------------------------------------------------*/
IQV_DLL_API unsigned int IQV_result_num_elements (IQV_HNDL_TYPE *hndl,
                                         char *parameter);
/*===========================================================================
   Description:
        This function is used to determine the number of elements contained
        in a vector result (array). This function is most useful for 
        allocating memory for a particular result when the data type is known. 
     
        For example, to retrieve the number of elements in IQV_RES_OFDM_PSDU_RX
        use:
        
        int num_elements = IQV_result_num_elements(hndl,IQV_RES_OFDM_PSDU_RX);

   Input:
      hndl       handle to tester session
      parameter - the name of the parameter, of which the length of is desired
         
   Output:
      result    - the number of elements found in the vector (array)
   
   Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_UNKNOWN_PARM

    ===========================================================================*/


/*---------------------------------------------------------------------------*/
IQV_DLL_API unsigned int IQV_result_num_capture_elements (IQV_HNDL_TYPE *hndl);
/*===========================================================================
   Description:
        This function may be used to determine the number of elements (samples)
        contained in the current capture buffer. 

   Input:
      hndl       handle to tester session
         
   Output:
      result    - the number of elements found in the vector (array)
   
   Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL

    ===========================================================================*/


/*---------------------------------------------------------------------------*/
IQV_DLL_API unsigned int IQV_result_element_size (IQV_HNDL_TYPE *hndl,
                             char *parameter);
/*===========================================================================
   Description:
        This function is used to determine the size (per element) contained
        in a result. This function is most useful for allocating memory for 
        a particular result when the number of elements is known. 

        For example, to retrieve the size of each element returned in  
        IQV_RES_OFDM_PSDU_RX use:
        
        int element_size = IQV_result_element_size(hndl,IQV_RES_OFDM_PSDU_RX);
     
   Input:
      hndl       handle to tester session
      parameter - the name of the parameter, of which the element size is 
                  desired         
   Output:
      result    - the element size of the result
   
   Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_UNKNOWN_PARM
  ===========================================================================*/


/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_get_bits_as_text (IQV_HNDL_TYPE *hndl,                                    
                          unsigned int scenario,
                          char *psdu_buffer,
                          int buff_len,
                          char *parameter);
/*===========================================================================
   Description:
       This function is used to retrieve a NULL terminated ASCII character 
       string (1s/0s) representation of binary data (such as 
       IQV_RES_OFDM_PSDU_RX). One should take care to allocate enough memory 
       for psdu_buffer, otherwise there may be missing elements. 

       One (1) is represented by ASCII character '1'. 
       Zero (0) is represented by ASCII character '0'.

       For example, one could use IQV_res_num_elements to discover how many
       elements there are in the array, before allocating memory and retrieve 
       the data: 
    
       int num_elements = IQV_result_num_elements(hndl, IQV_RES_OFDM_PSDU_RX);
       
       char *psdu;
       psdu = (char *)malloc(num_elements*sizeof(char)+1);
       if (psdu)
       {           
            int rtn_val = IQV_get_bits_as_text (hndl, 0, psdu, num_elements, 
                                                 IQV_RES_OFDM_PSDU_RX);
            printf ("PSDU String: %s\n", psdu);
       }

   Input:
      hndl         handle to tester session
      scenario     the ID of the scenario to work with
      psdu_buffer - pointer to a buffer where the text string will be copied
      buff_len     size of the buffer including room for the null terminator      
      parameter   - the name of the parameter to retrieve as a bit string
       
   Output:
      psdu_buffer - text string is copied to psdu_buffer
   
   Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_UNKNOWN_PARM
  
  ===========================================================================*/


/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_get_bits_as_ints (IQV_HNDL_TYPE *hndl,                                    
                          unsigned int scenario,
                          int *psdu_buffer,
                          int buff_len,
                          char *parameter);
/*===========================================================================
     Description:
       This function is used to retrieve an integer array representation 
       of binary data (such as IQV_RES_OFDM_PSDU_RX). One should take care to
       allocate enough memory for psdu_buffer, otherwise there may be missing
       elements. 

       One (1) is represented by the integer value 1.
       Zero (1) is represented by the integer value 0.

       For example, one could use IQV_res_num_elements to discover how many
       elements there are in the array, before allocating memory and retrieve 
       the data: 
    
       int *psdu;
       int num_elements = IQV_result_num_elements(hndl, IQV_RES_OFDM_PSDU_RX);
       psdu = (int *)malloc(num_elements*sizeof(int));
       
       if (psdu)
       {
            int rtn_val = IQV_get_bits_as_ints (hndl, 0, psdu, num_elements, 
                                                 IQV_RES_OFDM_PSDU_RX);
       }
   Input:
      hndl         handle to tester session
      scenario     the ID of the scenario to work with
      psdu_buffer - pointer to a buffer where the values will be copied
      buff_len     size of the buffer in terms of number of integer elements
      parameter   - the name of the parameter to retrieve as a bit string
       
   Output:
      psdu_buffer - values are copied to psdu_buffer
   
   Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_UNKNOWN_PARM
  ===========================================================================*/


/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_get_bits_as_hex (IQV_HNDL_TYPE *hndl,                                    
                          unsigned int scenario,
                          unsigned char *psdu_buffer,
                          int buff_len,
                          char *parameter);
/*===========================================================================
      Description:
       This function is used to retrieve a hexadecimal string representation 
       of binary data (such as IQV_RES_OFDM_PSDU_RX). One should take care to
       allocate enough memory for psdu_buffer, otherwise there may be missing
       elements. 

       For example, one could use IQV_res_num_elements to discover how many
       elements there are in the array, before allocating memory and retrieve 
       the data. In this case, the number of elements should be divided by 8, 
       as there would be eight bits per byte. One should make sure to round 
       up the value to the nearest number of bytes. Also, note that this 
       value will not be returned as NULL terminated. 
    
       int num_elements = IQV_result_num_elements(hndl, IQV_RES_OFDM_PSDU_RX);
       psdu = (char *)malloc(num_elements*sizeof(char)/8);
       
       if (psdu)
       {
            int rtn_val = IQV_get_bits_as_hex (hndl, 0, psdu, num_elements, 
                                                 IQV_RES_OFDM_PSDU_RX);
       }

   Input:
      hndl         handle to tester session
      scenario     the ID of the scenario to work with
      psdu_buffer - pointer to a buffer where the hexadecimal values will 
                    be copied
      buff_len     size of the buffer of elements      
      parameter   - the name of the parameter to retrieve as a bit string
       
   Output:
      psdu_buffer - hexadecimal values are copied to psdu_buffer
   
   Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_UNKNOWN_PARM
  ===========================================================================*/


/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_token_set_id (unsigned int token, 
                                  unsigned int token_id);
/*=============================================================================
   Description:
      This function is used to set the ID of the application code instance
      using the tokens once. Each application utilizing the tester should have
      a unique token ID. For instance, the token ID for an application can be
      set to the value 1, and another application could use the value 2. 
      The value 0 is an illegal token ID value.
      
   Input:
      token    - the token to receive the release status for 
                 (see enumeration IQV_TOKEN_ENUM for valid values)
      token_id - the token ID value to use for token. The value should not be
                 set to 0.   
   
   Output:
      Was a problem detected (FALSE) or not (TRUE).
      
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_TOKEN
  
===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_token_retrieve (IQV_HNDL_TYPE *hndl, 
                                    unsigned int   scenario,
                                    unsigned int   token,
                                    unsigned int   probe_time);
/*=============================================================================
   Description:
      This function will attempt to retrieve the token from the tester. It will 
      wait indefinitely until the token becomes available. It will attempt to 
      retrieve the token every probe_time msec. This time should not be too
      short, as the load on the tester will increase.
      
      After retrieving the token, the tester settings will be retrieved and
      stored in the selected scenario.
      
      NOTE: Since this is a blocking function, it is important that all 
            applications utilizing the token handling always release the token
            after use. If this is not done, then other applications trying to 
            retrieve the token will block.
      
   Input:
      hndl        handle to tester session
      scenario   - the scenario to store to when retrieving the tester settings
      token      - the token to retrieve
      probe_time - sleep time between each attempt to get the token
      
   Output:
      Was the token retrieved (TRUE) or token invalid (FALSE)
      
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_TOKEN
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_token_release (IQV_HNDL_TYPE *hndl, 
                                   unsigned int   token);
/*=============================================================================
   Description:
      This function releases the token. It will only release it if the token is 
      belonging to the application before attempting to release the token.
      
      If the token has already been released or is belonging to another 
      application, the function will not do anything.
      
   Input:
      hndl        handle to tester session
      token      - the token to release
      
   Output:
      Was the token valid (TRUE) or invalid (FALSE)
      
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_TOKEN      
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_token_release_problem (unsigned int token);
/*=============================================================================
   Description:
      NOTE: This function is not supported in this release. 

      This function returns the status of the previous token that was released 
      by the function IQV_token_release. If the token didn't have the expected 
      value when being released, the function will return TRUE. If the token 
      had the expected value when IQV_token_release was called, the function 
      will return FALSE.
      
   Input:
      token - the token to receive the release status for
              (see enumeration IQV_TOKEN_ENUM for valid values)
   
   Output:
      Was a problem detected (TRUE) or not (FALSE).
      
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_TOKEN
  
  ===========================================================================*/


/*---------------------------------------------------------------------------*/
IQV_DLL_API void IQV_token_reset (IQV_HNDL_TYPE *hndl);
/*=============================================================================
   Description:
      This function is used to reset the tester in regards to the token
      handling. This can be used to get the token out of the "hang" state if 
      an application having the token crashes, or otherwise stops responding.
      
   Input:
       hndl  handle to tester session

   Output:
        None
      
      
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_TOKEN
  
  ===========================================================================*/


/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_token_retrieve_timeout (IQV_HNDL_TYPE *hndl, 
                                            unsigned int   scenario,
                                            unsigned int   token,
                                            unsigned int   probe_time,
                                            unsigned int   timeout);
/*=============================================================================
    Description:
      This function will attempt to retrieve the token from the tester. It will 
      wait timeout milliseconds for the token to become available. It will 
      attempt to retrieve the token every probe_time msec. This time should not 
      be too short, as the load on the tester will increase. If the token is 
      not retrieved from the tester before timeout milliseconds, IQV_err() will 
      be set to IQV_ERR_TIMEOUT. If the user sets timeout to zero, a check for
      timeout will not occur, thus it would be the same as calling 
      IQV_token_retrieve.

      
      After retrieving the token, the tester settings will be retrieved and
      stored in the selected scenario.
      
      NOTE: Since this is a blocking function, it is important that all 
            applications utilizing the token handling always release the token
            after use. If this is not done, then other applications trying to 
            retrieve the token will block.
      
   Input:
      hndl        handle to tester session
      scenario   - the scenario to store to when retrieving the tester settings
      token      - the token to retrieve
      probe_time - sleep time between each attempt to get the token
      timeout    - time in milliseconds to wait for retrieving the token
      
   Output:
      Was the token retrieved (TRUE) or token invalid (FALSE)
      
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_TOKEN
      - IQV_ERR_TIMEOUT
  
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API double IQV_misc_cmd (IQV_HNDL_TYPE *hndl, 
                                 unsigned int   scenario, 
                                 char          *cmd,
                                 double         input);
/*===========================================================================
   Description:
     This function is used to send generic commands to the DLL sub-system.
     At present the command can be used to read the serial number - see
     file IQview_defines.h.
     
     The error level will be set to IQV_ERR_INVALID_HNDL, if the handle 
     supplied by the Test Suite is not recognized. Similarly, the error level 
     will be set to IQV_ERR_INVALID_SCENARIO, if the scenario ID supplied is 
     not recognized. 
     
     Use IQV_result to read the serial number as in the following example:
     
       IQV_RESULT_TYPE res;
       IQV_misc_cmd (hndl, MY_SCENARIO_0, IQV_RES_SERIAL_NO, 0);
       IQV_result (hndl, MY_SCENARIO_0, IQV_RES_SERIAL_NO, &res);
       printf ("Serial number: %s", res.desc);    
     
   Input:
      hndl        handle to tester session
      scenario    the ID of the scenario to work with
      cmd        - name of the generic command
      input      - input parameter for the generic command
   
   Output:
      result    - double value returned by the generic command used 
                  (default. value is 0).
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
      - IQV_ERR_INVALID_SCENARIO
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API double IQV_misc_multi_cmd_1 (IQV_HNDL_TYPE *hndl, 
                                         char          *cmd,
                                         IQV_MISC_TYPE *parm1);
/*===========================================================================
   Description:
     This function is used to send generic multi purpose parameters to the 
     DLL sub-system. This function allows for one (1) input parameter and one 
     (1) output parameter.
     
     At present, no specific command have been defined.
     
     The error level will be set to IQV_ERR_INVALID_HNDL, if the handle 
     supplied by the Test Suite is not recognized.
     
     Encode the parameters as in the following examples:

     Example for integer parameter passing:

       IQV_MISC_TYPE parm;
       int           integer;
       ...
       parm.type     = IQV_MISC_INT;
       parm.ptr_real = &integer;
       IQV_misc_multi_cmd_1(hndl, "Example Integer", &parm);    

     Example for real parameter passing:

       IQV_MISC_TYPE parm;
       double        real;
       ...
       parm.type     = IQV_MISC_REAL;
       parm.ptr_real = &real;
       IQV_misc_multi_cmd_1(hndl, "Example Real Number", &parm);    

     Example for complex parameter passing:

       IQV_MISC_TYPE parm;
       double        real;
       double        imag;
       ...
       parm.type     = IQV_MISC_COMPLEX;
       parm.ptr_real = &real;
       parm.ptr_imag = &imag;
       IQV_misc_multi_cmd_1(hndl, "Example Complex Number", &parm);    

     Example for textual parameter passing:

       IQV_MISC_TYPE parm;
       ...
       parm.type     = IQV_MISC_TXT;
       parm.ptr_real = &txt;   // txt contains a null-terminated text string.
       IQV_misc_multi_cmd_1(hndl, "Example Text String", &parm);    
       
     Example for real array parameter passing:

       IQV_MISC_TYPE parm;
       ...
       parm.type     = IQV_MISC_COMPLEX_ARRAY;
       parm.ptr_real = &real_array; // real array contains x elements.
       parm.ptr_imag = NULL;        // Imaginary part not used.
       parm.len      = x;
       IQV_misc_multi_cmd_1(hndl, "Example Real Array", &parm);    

     Example for complex array parameter passing:

       IQV_MISC_TYPE parm;
       ...
       parm.type     = IQV_MISC_COMPLEX_ARRAY;
       parm.ptr_real = &real_array; // real array contains x elements.
       parm.ptr_imag = &imag_array; // imaginary array contains x elements.
       parm.len      = x;
       IQV_misc_multi_cmd_1(hndl, "Example Complex Array", &parm);    
    
   Input:
      hndl        handle to tester session
      cmd        - name of the generic command
      parm1      - input parameter for the generic command
   
   Output:
      result    - double value returned by the generic command used 
                  (default. value is 0).
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
  
  ===========================================================================*/

/*---------------------------------------------------------------------------*/
IQV_DLL_API double IQV_misc_multi_cmd_5 (IQV_HNDL_TYPE *hndl, 
                                         char          *cmd,
                                         IQV_MISC_TYPE *parm1,
                                         IQV_MISC_TYPE *parm2,
                                         IQV_MISC_TYPE *parm3,
                                         IQV_MISC_TYPE *parm4,
                                         IQV_MISC_TYPE *parm5);
/*===========================================================================
   Description:
      Same description as for IQV_misc_multi_cmd_1. Set unused parameters to
      NULL.
    
   Input:
      hndl        handle to tester session
      cmd        - name of the generic command
      parm1-5    - input parameters for the generic command
   
   Output:
      result    - double value returned by the generic command used 
                  (default. value is 0).
      Typical error levels (can be read using the function IQV_err):
      - IQV_ERR_OK
      - IQV_ERR_INVALID_HNDL
  
  ===========================================================================*/

/*===========================================================================
  Error Level handling function
  ===========================================================================*/
  
/*---------------------------------------------------------------------------*/
IQV_DLL_API int IQV_err (void);
/*===========================================================================
   Description:
      This function is used to return the latest error level.
            
   Input:
      None
   
   Output:
      error leve
  
  ===========================================================================*/

/*------------------------------------------------------------------- End ---*/
#endif /* IQVIEW_H */