//////////////////////////////////////////////////////////////////////
// $RCSfile: GatherSloginInfo.cpp,v $
// $Revision: 1.1 $
// $Date: 2003/09/26 22:48:33 $
// $Author: doomer $
//
// Copyright (c) 2001 Agilent Technologies.  All rights reserved.
//////////////////////////////////////////////////////////////////////

#include <qapplication.h>
#include <qmessagebox.h> 
#include "ServiceLogonImpl.h"
#include "StrUtils.h"
#include "SlogonDlgInterface.h"
#include <windows.h>
#include <ntsecapi.h>
#include <lm.h>
#include <vector>

char *g_appName = 0;
QWidget *g_mainWidget = 0;

void DisplayLastError(const char *preamble, long lastError)
{
	LPVOID lpMsgBuf = 0;
	QString msg(preamble);
	msg += '\n';
	
	if (FormatMessage( 
		FORMAT_MESSAGE_ALLOCATE_BUFFER | 
		FORMAT_MESSAGE_FROM_SYSTEM | 
		FORMAT_MESSAGE_IGNORE_INSERTS,
		NULL,
		lastError,
		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
		(LPTSTR) &lpMsgBuf,
		0,
		NULL ))
	{
		msg += (char*)lpMsgBuf;
	}
	
	QMessageBox::critical(g_mainWidget, g_appName, msg);

	if(lpMsgBuf){
		LocalFree( lpMsgBuf );	
	}
}

#ifndef UNICODE
#define UNICODE
#endif // UNICODE

NTSTATUS
OpenPolicy(
    LPWSTR ServerName,          // machine to open policy on (Unicode)
    DWORD DesiredAccess,        // desired access to policy
    PLSA_HANDLE PolicyHandle    // resultant policy handle
    );

BOOL
GetAccountSid(
    LPTSTR SystemName,          // where to lookup account
    LPTSTR AccountName,         // account of interest
    PSID *Sid                   // resultant buffer containing SID
    );

NTSTATUS
SetPrivilegeOnAccount(
    LSA_HANDLE PolicyHandle,    // open policy handle
    PSID AccountSid,            // SID to grant privilege to
    LPWSTR PrivilegeName,       // privilege to grant (Unicode)
    BOOL bEnable                // enable or disable
    );

void
InitLsaString(
    PLSA_UNICODE_STRING LsaString, // destination
    LPWSTR String                  // source (Unicode)
    );

void
DisplayNtStatus(
    LPSTR szAPI,                // pointer to function name (ANSI)
    NTSTATUS Status             // NTSTATUS error value
    );

//
// If you have the ddk, include ntstatus.h.
//
#ifndef STATUS_SUCCESS
#define STATUS_SUCCESS  ((NTSTATUS)0x00000000L)
#endif

bool GrantServiceRights(char *AccountName)
{
	bool result = false;
	LSA_HANDLE PolicyHandle;
    PSID pSid;

    NTSTATUS Status = OpenPolicy(
                0,                  // target machine
                POLICY_CREATE_ACCOUNT | POLICY_LOOKUP_NAMES,
                &PolicyHandle       // resultant policy handle
                );

    if(Status != STATUS_SUCCESS) {
        DisplayNtStatus("OpenPolicy", Status);
        return false;
    }

    //
    // Obtain the SID of the user/group.
    // Note that we could target a specific machine, but we don't.
    // Specifying NULL for target machine searches for the SID in the
    // following order: well-known, Built-in and local, primary domain,
    // trusted domains.
    //
    if(GetAccountSid(
            NULL,       // default lookup logic
            AccountName,// account to obtain SID
            &pSid       // buffer to allocate to contain resultant SID
            )) {
        //
        // We only grant the privilege if we succeeded in obtaining the
        // SID. We can actually add SIDs which cannot be looked up, but
        // looking up the SID is a good sanity check which is suitable for
        // most cases.

        //
        // Grant the SeServiceLogonRight to users represented by pSid.
        //
        Status = SetPrivilegeOnAccount(
                    PolicyHandle,           // policy handle
                    pSid,                   // SID to grant privilege
                    L"SeServiceLogonRight", // Unicode privilege
                    true                    // enable the privilege
                    );

        if(Status == STATUS_SUCCESS)
            result = true;
        else
            DisplayNtStatus("SetPrivilegeOnAccount", Status);
    }
    else {
        //
        // Error obtaining SID.
        //
        DisplayLastError("GetAccountSid", GetLastError());
    }

    //
    // Close the policy handle.
    //
    LsaClose(PolicyHandle);

    //
    // Free memory allocated for SID.
    //
    if(pSid != NULL) HeapFree(GetProcessHeap(), 0, pSid);

    return result;
}

/*++
This function attempts to obtain a SID representing the supplied
account on the supplied system.

If the function succeeds, the return value is TRUE. A buffer is
allocated which contains the SID representing the supplied account.
This buffer should be freed when it is no longer needed by calling
HeapFree(GetProcessHeap(), 0, buffer)

If the function fails, the return value is FALSE. Call GetLastError()
to obtain extended error information.

Scott Field (sfield)    12-Jul-95
--*/

BOOL
GetAccountSid(
    LPTSTR SystemName,
    LPTSTR AccountName,
    PSID *Sid
    )
{
    LPTSTR ReferencedDomain=NULL;
    DWORD cbSid=128;    // initial allocation attempt
    DWORD cchReferencedDomain=16; // initial allocation size
    SID_NAME_USE peUse;
    BOOL bSuccess=FALSE; // assume this function will fail

    __try {

    //
    // initial memory allocations
    //
    *Sid = (PSID)HeapAlloc(GetProcessHeap(), 0, cbSid);

    if(*Sid == NULL) __leave;

    ReferencedDomain = (LPTSTR)HeapAlloc(
                    GetProcessHeap(),
                    0,
                    cchReferencedDomain * sizeof(TCHAR)
                    );

    if(ReferencedDomain == NULL) __leave;

    //
    // Obtain the SID of the specified account on the specified system.
    //
    while(!LookupAccountName(
                    SystemName,         // machine to lookup account on
                    AccountName,        // account to lookup
                    *Sid,               // SID of interest
                    &cbSid,             // size of SID
                    ReferencedDomain,   // domain account was found on
                    &cchReferencedDomain,
                    &peUse
                    )) {
        if (GetLastError() == ERROR_INSUFFICIENT_BUFFER) {
            //
            // reallocate memory
            //
            *Sid = (PSID)HeapReAlloc(
                        GetProcessHeap(),
                        0,
                        *Sid,
                        cbSid
                        );
            if(*Sid == NULL) __leave;

            ReferencedDomain = (LPTSTR)HeapReAlloc(
                        GetProcessHeap(),
                        0,
                        ReferencedDomain,
                        cchReferencedDomain * sizeof(TCHAR)
                        );
            if(ReferencedDomain == NULL) __leave;
        }
        else __leave;
    }

    //
    // Indicate success.
    //
    bSuccess = TRUE;

    } // try
    __finally {

    //
    // Cleanup and indicate failure, if appropriate.
    //

    HeapFree(GetProcessHeap(), 0, ReferencedDomain);

    if(!bSuccess) {
        if(*Sid != NULL) {
            HeapFree(GetProcessHeap(), 0, *Sid);
            *Sid = NULL;
        }
    }

    } // finally

    return bSuccess;
}

NTSTATUS
SetPrivilegeOnAccount(
    LSA_HANDLE PolicyHandle,    // open policy handle
    PSID AccountSid,            // SID to grant privilege to
    LPWSTR PrivilegeName,       // privilege to grant (Unicode)
    BOOL bEnable                // enable or disable
    )
{
    LSA_UNICODE_STRING PrivilegeString;

    //
    // Create a LSA_UNICODE_STRING for the privilege name.
    //
    InitLsaString(&PrivilegeString, PrivilegeName);

    //
    // grant or revoke the privilege, accordingly
    //
    if(bEnable) {
        return LsaAddAccountRights(
                PolicyHandle,       // open policy handle
                AccountSid,         // target SID
                &PrivilegeString,   // privileges
                1                   // privilege count
                );
    }
    else {
        return LsaRemoveAccountRights(
                PolicyHandle,       // open policy handle
                AccountSid,         // target SID
                FALSE,              // do not disable all rights
                &PrivilegeString,   // privileges
                1                   // privilege count
                );
    }
}

void
InitLsaString(
    PLSA_UNICODE_STRING LsaString,
    LPWSTR String
    )
{
    DWORD StringLength;

    if(String == NULL) {
        LsaString->Buffer = NULL;
        LsaString->Length = 0;
        LsaString->MaximumLength = 0;
        return;
    }

    StringLength = lstrlenW(String);
    LsaString->Buffer = String;
    LsaString->Length = (USHORT) StringLength * sizeof(WCHAR);
    LsaString->MaximumLength=(USHORT)(StringLength+1) * sizeof(WCHAR);
}

NTSTATUS
OpenPolicy(
    LPWSTR ServerName,
    DWORD DesiredAccess,
    PLSA_HANDLE PolicyHandle
    )
{
    LSA_OBJECT_ATTRIBUTES ObjectAttributes;
    LSA_UNICODE_STRING ServerString;
    PLSA_UNICODE_STRING Server;

    //
    // Always initialize the object attributes to all zeroes.
    //
    ZeroMemory(&ObjectAttributes, sizeof(ObjectAttributes));

    if (ServerName != NULL) {
        //
        // Make a LSA_UNICODE_STRING out of the LPWSTR passed in
        //
        InitLsaString(&ServerString, ServerName);
        Server = &ServerString;
    } else {
        Server = NULL;
    }

    //
    // Attempt to open the policy.
    //
    return LsaOpenPolicy(
                Server,
                &ObjectAttributes,
                DesiredAccess,
                PolicyHandle
                );
}

void
DisplayNtStatus(
    LPSTR szAPI,
    NTSTATUS Status
    )
{
    //
    // Convert the NTSTATUS to Winerror. Then call DisplayLastError().
    //
    DisplayLastError(szAPI, LsaNtStatusToWinError(Status));
}

static class WinsockInit{
    bool m_initialized;

    public:
        WinsockInit() : m_initialized(false) {}
        ~WinsockInit() {
            WSACleanup();
        }

        bool Init(){
            if( ! m_initialized){
                const unsigned char major = 1;
                const unsigned char minor = 0;

                WORD wVersionRequested = MAKEWORD( major, minor );
                WSADATA wsaData;
                int err = WSAStartup( wVersionRequested, &wsaData );

                if ( ! err){
                    m_initialized = true;
                }
            }

            return m_initialized;
        }

} s_winSockInit;

extern BOOL WINAPI SSPLogonUser(LPTSTR szDomain, LPTSTR szUser,
                                LPTSTR szPassword, HANDLE *htok);

bool TokenIsAdmin(HANDLE htok)
{
	bool result = false;
	
    SID_IDENTIFIER_AUTHORITY ntauth = SECURITY_NT_AUTHORITY;
    void* psidAdmin = 0;
    AllocateAndInitializeSid( &ntauth, 2,
                              SECURITY_BUILTIN_DOMAIN_RID,
                              DOMAIN_ALIAS_RID_ADMINS,
                              0, 0, 0, 0, 0, 0, &psidAdmin );
    DWORD cb = 0;
    GetTokenInformation( htok, TokenGroups,
                         0, 0, &cb );
    TOKEN_GROUPS* ptg = (TOKEN_GROUPS*) malloc(cb);
    GetTokenInformation( htok, TokenGroups,
                         ptg, cb, &cb );

    DWORD groupIndex = ptg->GroupCount;

    for ( DWORD i = 0; i < ptg->GroupCount; ++i ){
        if ( EqualSid( psidAdmin, ptg->Groups[i].Sid ) ){
            groupIndex = i;
            break;
        }
    }

    result = groupIndex != ptg->GroupCount;

    free( ptg );
    FreeSid( psidAdmin );

	return result;
}

std::string TheDomainName(const std::string &accountName)
{
    std::string result;

    if((accountName.length()) && (accountName.find('\\') != std::string::npos)){
        std::vector<std::string> splitResult =
            MsmtSrvcStrUtils::Utilities::Split::SplitString(accountName, '\\');
        if(splitResult[0] != "."){
            result = splitResult[0];
        }        
    }
    
    return result;
}

std::string TheAccountName(const std::string &accountName)
{
    std::string result(accountName);

    if((accountName.length()) && (accountName.find('\\') != std::string::npos)){
        std::vector<std::string> splitResult =
            MsmtSrvcStrUtils::Utilities::Split::SplitString(accountName, '\\');
        result = splitResult[splitResult.size() - 1];
    }

    return result;
}

bool AccountIsAdmin(const std::string &accountName, const std::string &domainName,
                    const std::string &password)
{
    bool result = false;
    HANDLE htok;

    std::string theDomain(domainName);
    if( ! theDomain.length()){
        const int nameLen = 255;
        char hostName[nameLen + 1];
        if( ! s_winSockInit.Init()){
            return result;
        }
        
        if( ! gethostname(hostName, nameLen)){
            theDomain = hostName;
        }
    }

    if(SSPLogonUser((char*)theDomain.c_str(), (char*)accountName.c_str(),
                    (char*)password.c_str(), &htok)){
        if(htok){
            if(TokenIsAdmin(htok)){
                result = true;
            }
            
            CloseHandle(htok);
        }
    }

    return result;
}

void GatherSloginInfo(SLoginOptions *options)
{
    static char *argv = "Connection Manager Service Account";
	g_appName = argv;
    int argc = 1;

	QApplication app( argc, &argv );

	ServiceLogonImpl dialog( 0, 0, TRUE );
	app.setMainWidget(&dialog);
	g_mainWidget = &dialog;

	bool keepGoing = true;
    bool serviceRightsGranted = false;

	while(keepGoing){
		if(dialog.exec()){
			if(dialog.ThisAccountChecked()){
                std::string theAccount = dialog.AccountName();
				std::string account = TheAccountName(theAccount);
				if(account.length() > 0){
					std::string password = dialog.ValidatedPassword();
					if(password.length() > 0){
						password = dialog.ValidatedPassword();
                        if( ! serviceRightsGranted){
                            if( ! GrantServiceRights(const_cast<char*>(account.c_str()))){
                                DisplayLastError("Could not grant the account the "
                                    "Logon As Service right", GetLastError());
                            }
							else{
								serviceRightsGranted = true;
							}
                        }

                        if(AccountIsAdmin(account, TheDomainName(theAccount),
                                          password)){
                            keepGoing = false;
                            // register the service using the account name
                            // and password
                            options->loginAsLocalSystem = false;
                            options->interactWithDesktop = false;
                            options->account = theAccount;
                            options->password = dialog.ValidatedPassword();
                        }
                        else{
                            long le = GetLastError();
                            if(le != NO_ERROR){
                                DisplayLastError("Could not validate login/password "
                                    "combination as an Administrator", le);
                            }
                            else{
                                QMessageBox::critical(&dialog, g_appName, "Could not validate login/password "
                                    "combination as an Administrator");
                            }
                        }
					}
					else{
						QMessageBox::critical(&dialog, g_appName,
							"Password and confirmation must have an entry and "
							"the entries must match");
					}
				}
				else{
					QMessageBox::critical(&dialog, g_appName,
						"Please enter an account name");
				}
			}
			else{
				if(dialog.InteractWithDesktopChecked()){
					// register the service under the local system account
					// with access to the desktop
                    options->loginAsLocalSystem = true;
                    options->interactWithDesktop = true;

					keepGoing = false;
				}
				else{
					// register the service under the local system account
					// without access to the desktop
                    options->loginAsLocalSystem = true;
                    options->interactWithDesktop = false;

					keepGoing = false;
				}
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////
// $Log: GatherSloginInfo.cpp,v $
// Revision 1.1  2003/09/26 22:48:33  doomer
// fix for Instrument_Links.255
// Added Files:
//  	AccountSelectImpl.h AccountSelectionImpl.cpp
//  	GatherSloginInfo.cpp ServiceLogonImpl.cpp ServiceLogonImpl.h
//  	SlogonDlgInterface.h StrUtils.cpp StrUtils.h
//  	VerifyLoginInfo.cpp accountselection.cpp accountselection.h
//  	accountselection.ui helpbox.cpp helpbox.h helpbox.ui main.cpp
//  	make-defs moc_accountselection.cpp moc_helpbox.cpp
//  	moc_servicelogon.cpp qt320.tar.gz servicelogon.cpp
//  	servicelogon.h servicelogon.ui
//
//////////////////////////////////////////////////////////////////////
