// ****************************************************************************
//
// File Name:    demo.c
//
// Description: Main demonstration program. 
//              Used to give an HMS serial commands.
//
//
//  ORIGINATOR: Escort Memory Systems
//
//  HISTORY
//    who     when     what
// -------- --------- ----------------------------------------------------
//
// ****************************************************************************

#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>     // Lower case conversion and toupper.
#include <dos.h>
#include "typedef.h"   // General typedef and defines.
#include "handheld.h"  // MUX enabled and MUX address defines.
#include "comm.h"
#include "hms800.h"
#include "preform.h"

#define TOO_MANY_ERRORS      10  // Too many read errors on this tag.
#define TAG_IO_NUM     mem_size  // set to MEM_SIZE for an entire tag.

static int serial_port;    // Remember the port number being used, to change baud rates.
static BYTE UART_control;  // Copy of the current UART control register value.

static BYTE cur_value;     // The current value being written to each byte in the tag.
static BYTE hold_resp;     // Hold the input character while the screen is cleared.

static WORD cont_start_add = 0; // Continual read mode start address
static WORD cont_size = 0;      // Continual read mode data SIZE, number of bytes.

// ****************************************************************************
// Display a Block Read Response. This is for both normal Block Reads and
// in continual Block Read mode a tag is found.
//
// Input:        Data size, the number of bytes expected.
//               Tag address of the bytes that where requested.
//               BOOLEAN - is this a continual block read?
//                         TRUE - receiving from continual block read.
//                         FALSE - receiving from normal Block Wroite. 
// Return:       NONE.
// Side effects: NONE.
static void disp_BLReadRx(WORD data_sz, WORD tag_addr, BYTE cont_bl)
{
   BYTE status;
   BYTE *rx_buf;

   rx_buf = (BYTE*) malloc(data_sz);
    if (cont_bl)
      status = ContBLReadRx(rx_buf, data_sz);
   else
      status = BLReadRx(rx_buf, data_sz);

   if ((status EQ OP_OK)) //print out receiving data 
      display_data(rx_buf, data_sz, tag_addr);
   else 
      printf("\n\nBlock Read failure with Error Code [%02X].\n", status);
   free(rx_buf);
}

// ****************************************************************************
// Initial initialize serial port
//
// Input:        NONE, implied global value passed in serial_port.
// Return:       NONE.
// Side effects: Initilizes the COM port specified by serial_port..
static void InitCom (void)
{
    printf("\nVersion %s, using COM%1d at 9600 baud rate. \n", VERSION_STR, serial_port);

    comm_open(serial_port, 9600, UART_control = WORD_EIGHT);
    comm_flush();
}

// ****************************************************************************
// Send the enter (or exit) continual read mode command the an HMS827.
//
// Input:        NONE.
// Return:       NONE.
// Side effects: NONE.
static void inf_BlockRead(void)
{
   BYTE status;

   clrscr();
   //Get start address and data size to read
   for(;;) 
   {
      printf("\nInput data size (no more then %d):", max_data_pkt);
      if ((cont_size = inp_num(max_data_pkt)) EQ ENTER_ALONE)
         cont_size = 0;                 // ENTER alone, then assume size of zero.
      if (cont_size EQ TOO_BIG_VAL)      // User trying to ESCape?
         return;                         // Yes, let the user out. 

      if (cont_size)    // If enabling continual read.
      {
         printf("\nInput continual read start address[0..%u]: ",
            mem_size - cont_size - 1);
         cont_start_add = inp_num(mem_size - cont_size - 1); // Enter a number in decimal format.
         if ((cont_start_add EQ ENTER_ALONE) OR (cont_size EQ TOO_BIG_VAL))
            return;                         //Let the user out. 
      } else // Continual read is being disabled.
         cont_start_add = 0; // NO byte count and NO start address (0).
      break;
   }
  
   printf("\nSending command to tag.\n");
   ContReadTx(cont_start_add, cont_size); 
   status = ContReadRx();

   if (status EQ OP_OK)
   {
      if (cont_size)    // If enabling continual read.
         printf("HMS827 entered continual read mode.\n");
      else
         printf("HMS827 not continually reading.\n");
   }
   else
      print_code(status);

   printf("\nPress any key to go back to menu. \n");
   getch();
}

//*****************************************************************************
// Change the HMS serial baud rate.  Send a command to the device and change
// this PC's baud rate.  Both this device and the HMS development system
// should change their baud rate, to what is request here.
//
// Input:        NONE.
// Return:       NONE.
// Side effects: Alters this machine's baud rate.
static void ChangeBaud(void) 
{
   WORD baud_value;  // Hold the desired baud rate value.

   clrscr();
   do
   {
      printf("\n\nSet baud rate to:\n");
      printf(" 0: 1200 \n");
      printf(" 1: 2400\n");
      printf(" 2: 4800\n");
      printf(" 3: 9600\n");
      printf(" 4: 19200\n");
      printf(" 5: 38400\n");
      printf(" 6: 57600\n");
      printf(" 7: 115200\n");

      while(NOT kbhit());   // Wait for key input.
      switch  (getch())
      {
      case '0':
         baud_value = 12;   // Set the baud rate value for dobaud(), in comm.c.
         break;

      case '1':
         baud_value = 24;   // Set the baud rate value for dobaud(), in comm.c.
         break;

      case '2':
         baud_value = 48;   // Set the baud rate value for dobaud(), in comm.c.
         break;

      case '3':
         baud_value = 96;   // Set the baud rate value for dobaud(), in comm.c.
         break;

      case '4':
         baud_value = 192;   // Set the baud rate value for dobaud(), in comm.c.
         break;

      case '5':
         baud_value = 384;   // Set the baud rate value for dobaud(), in comm.c.
         break;

      case '6':
         baud_value = 576;   // Set the baud rate value for dobaud(), in comm.c.
         break;

      case '7':
         baud_value = 1152;  // Set the baud rate value for dobaud(), in comm.c.
         break;

      case ESC:
         return;

      default:
         printf("NO! \7 Enter a number in the list (dummy).\n");
         baud_value = 0;  // No valid entry, thus no baud rate value.
         break;
         ;
      }
   } while (NOT baud_value);  // Keep requesting new baud rate until entered.      
   clrscr();
   BaudrateTx(baud_value);
   delay(500);  // Wait a half a second, for the transmit buffer to fully empty.
   comm_open(serial_port, (long )baud_value * 100, UART_control);
   comm_flush();
   delay(100);  // Wait for the transmit buffer setup to take effect.
   printf("\n\n Baud rate in port %1d is changed to ", serial_port); 
   if (baud_value EQ 1152)
       printf("115200.\n");
   else
       printf("%6u.\n", baud_value * 100);
   getch();    // Wait with baud rate change displayed.
}

//*****************************************************************************
// Choose a command for the antenna.
//
// Input:        NONE.
// Return:       NONE.
// Side effects: NONE.
static void req_HMScommand(void) 
{
   WORD bytes_printed;  // Keep a count if mystery data bytes for display.

   for(;;)
   {
      printf("\nCommand for");
      print_tag_type();   // Print tag type for these commands.
      printf("\n");
#if TEST_MUX32
      printf(" ---------> Using MUX32 address %d <---------\n", now_mux_addr);
#endif      
      printf(" 0. Read Continually.\n");
      printf(" 1. Block Read.\n");
      printf(" 2. Block Write.\n");
      printf(" 3. Protected Block Write.\n");
      printf(" 4. Search Tag.\n");
      printf(" 5. Read tag ID.\n");
      printf(" 6. Clear Tag.\n");
      printf(" 7. Fill Tag.\n");
      printf(" 8. Check antenna.\n");
      printf(" 9. Configure Non-contiguous R/W.\n");
      printf(" A. Non-contiguous Read. Configuration assumed.\n");
      printf(" B. Non-contiguous write. Configuration assumed.\n");
      printf(" C. Change baud rate.\n");
      printf(" D. Infinite retry Block Reads.\n");
      printf(" E. Copy from a tag.\n");
      if ((start_adr NE TOO_BIG_VAL) AND (stop_adr NE TOO_BIG_VAL))
      {
         printf(" F. Write copied tag.\n");
         printf(" G. View copied data.\n");
      }
      printf("Selection ........   ");
      hold_resp = 0;
      while(NOT kbhit())
         if (comm_avail()) // Have received somthing. Is the continual read response?
         {
            if (cont_size)  // A pending continual read of some size active?
            {
               printf("\nJust got (from continual read at %d for %d bytes)\n",
                  cont_start_add, cont_size);
               disp_BLReadRx(cont_size, cont_start_add, TRUE); 
            } else
            {
               printf("\nJust received:\n");
               bytes_printed = 0;   // No data printed yet.
               do
               { 
                  while (comm_avail())
                  {
                     if (bytes_printed++ >= 15)
                     {
                        printf("\n");
                        bytes_printed = 0;
                     }
                     printf(" %2x,", (BYTE) comm_getc());
                  }
                  display_delay_sht();     // Wait for some more bytes.
               } while (comm_avail()); // Get all the data in one string.
               printf("\n");
            }
         }
      hold_resp = getch();   
      clrscr();

      switch(toupper(hold_resp))
      {
      case '0':
         ContinueRead();
         break;

      case '1':
         BlockRead();
         break;

      case '2':
         pro_BlockWrite(FALSE);
         break;

      case '3':
         pro_BlockWrite(TRUE);
         break;

      case '4':
         SearchTag();
         break;

      case '5':
         Get_tag_ID();
         break;

      case '6':
         ClearTag();
         break;

      case '7':
         FillTag();
         break;

      case '8':
         ant_attached();
         break;

      case '9':
          read_write_conf();
          break;

      case 'A':
          read_notcont();
          break;

      case 'B':
          write_notcont();
          break;

      case 'C':
         ChangeBaud();
         break;

      case 'D':
         inf_BlockRead();
         break;

      case 'E':
         copy_from_tag();
         break;

      case 'F':
         if ((start_adr NE TOO_BIG_VAL) AND (stop_adr NE TOO_BIG_VAL))
            copy_to_tag();
         break;

      case 'G':
         if ((start_adr NE TOO_BIG_VAL) AND (stop_adr NE TOO_BIG_VAL))
            disp_copied();
         break;

      case ESC:
         return;

      default:
         ;
      }

   clrscr();
   }
}

// ****************************************************************************
// The main routine.  Choose an antanna type.
//
// Input:        Command line options, like the COM port number.
// Return:       NONE.
// Side effects: NONE.
void main(argc, argv)
int argc;            // Agetc returns an integer not a char
char **argv;
{
#if TEST_MUX32
   WORD hold_adr;   // Hold the input MUX32 address, to verify it is not an ESC.
#endif
   
   if(argc <= 1)        // Always have at least one parameter?
      serial_port = 1;  // Default comport is port one.
   else
      serial_port = atoi(argv[1]); // Setup implied parameter. 
   
   if ((serial_port < MIN_COM) OR ((serial_port > MAX_COM)))
      {
      printf("Use comport between\n%u and %u only.\n", MIN_COM, MAX_COM);
      exit(1);
      }
   InitCom();

   cur_value = 0;       // For write to entire tag test. Start with a value.
   max_data_pkt = HMS827_MAX_DATA;
   mem_size = HMS_MEM_SIZE;
   antenna_type = HMS_FAMILY;
   for(;;)     // Loop forever, use exit command.
   {
      comm_flush();         // Clear out any residue from the command.
#if TEST_MUX32
      printf("Using MUX32 address %d \n\n", now_mux_addr);
#endif      
      printf("Data display format:\n");
      printf(" 0. Hexadecimal\n");
      printf(" 1. Decimal\n");
      printf(" 2. ASCII\n");
      printf(" 3. Binary\n");
#if TEST_MUX32
      printf(" 4: Set MUX32 address.\n");
#endif      
      printf(" X: Exit \n\n");
      printf("Select data display format: ");
      while(!kbhit());
      clrscr();

      switch(toupper(getch()))
      {
      case '0':
         display_type = HEX;
         req_HMScommand();
         break;

      case '1':
         display_type = DEC;
         req_HMScommand();
         break;

      case '2':
         display_type = ASCII;
         req_HMScommand();
         break;

      case '3':
         display_type = BIN;
         req_HMScommand();
         break;

#if TEST_MUX32
      case '4':
         printf("Input new MUX32 address: ");
         hold_adr = inp_num(BYTE_VALUE);  // Get BYTE value input for address.
         if ((hold_adr NE ENTER_ALONE) AND (hold_adr NE TOO_BIG_VAL))
            now_mux_addr = hold_adr;
         break;
#endif         

      case 'X':
         comm_close();
         exit(0);

      default:
         ;
      }
   }
}
