/* Copyright (C) 2009 by Black Cat Software
   Written by Shawn Fessenden
*/

#define STRICT
#define WIN32_LEAN_AND_MEAN

#include <windows.h>

// Timer id.
#define SKYEY_VOIDS 0x5b637900

// Error returns from StallGuard_GetLastError.
#define ERR_NO_ERROR			0
#define ERR_NO_SUBCLASS		1
#define ERR_NO_VEE				2
#define ERR_NO_UNSUBCLASS	3


// SINGLE INSTANCE data.
static HINSTANCE	ghInst;
static WNDPROC		gWndProc;

static UINT	guAclMs;				//milliseconds to wait before canceling accelerator incompletion.
static UINT	guCtxMs;				//milliseconds to wait before canceling context menu incompletion.
static BOOL	gbMonitor;			//turn stall guard monitoring on/off;   0 = off, 1 = on.
static BOOL	gbNotify;				//turn message box notification on/off; 0 = off, 1 = on.
static UINT	guLastError;		//last error.
static UINT	guCancelCount;	//number of times automatic cancel was sent.

static BOOL	gbGotEnterMenuLoop;
static BOOL gbWasContextMenu;


// CALLBACKS

// This function is called when the timer expires.
VOID CALLBACK TimerProc(HWND h, UINT uMsg, UINT idEvent, DWORD dwTime)
{
	// If we're still waiting for command completion...
	if(gbGotEnterMenuLoop) {
		
		// Increment the cancel count (for the curious).
		guCancelCount++;
		
		// And cancel the command by simulating an Escape keypress.
		PostMessage(h, WM_KEYDOWN, VK_ESCAPE, 0);
		PostMessage(h, WM_KEYUP, VK_ESCAPE, 0);
	}
	
	// Don't forget to kill the timer!
	KillTimer(h, SKYEY_VOIDS);
}	

// New window procedure. This function gets called before VEE for every message.
// It must be fast: it gets flooded with mouse messages whenever the mouse moves.
LRESULT CALLBACK WndProc(HWND h, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	// Only if we're interested in monitoring...
	if(gbMonitor) {
		
		// If this is a WM_EXITMENULOOP message then the
		// user is manipulating VEE and everything is ok.
		if(WM_EXITMENULOOP == uMsg) gbGotEnterMenuLoop = FALSE;
		
		// On the other hand, if it's WM_ENTERMENULOOP...
		else {
			if(WM_ENTERMENULOOP == uMsg) {
				
				// Note that we're now waiting for command completion & whether it's ALT key or right click.
				gbGotEnterMenuLoop = TRUE;
				gbWasContextMenu = (BOOL)wParam;
				
				// Wait specified time in milliseconds.
				SetTimer(h, SKYEY_VOIDS, gbWasContextMenu ? guCtxMs : guAclMs, TimerProc);
			}
		}
	}
	
	// In any case, always pass all messages on to VEE.
	return CallWindowProc(gWndProc, h, uMsg, wParam, lParam);
}


// This function collects top-level window data.
BOOL CALLBACK EnumWndProc(HWND h, LPARAM l)
{
	HWND *hp = (HWND*)l;
	*(hp + (UINT)*hp) = h;
	((UINT)*hp)++;
	return TRUE;
}


// UTILITY

// Returns TRUE if unknown string contains test string. Not case sensitive.
BOOL str_contains_str(char *unk, char *tst)
{
	int		len = lstrlen(tst);
	char	*p, *local;
	BOOL	ret = FALSE;

	local = (char *)GlobalAlloc(GPTR, (len + 1));
	for(p = unk; p < (unk + (lstrlen(unk) - lstrlen(tst))); p++) {
		lstrcpyn(local, p, len + 1);
		if(lstrcmpi(tst, local) == 0) {
			ret = TRUE;
			break;
		}
	}
	GlobalFree(local);
	return ret;
}

// Attempts to find VEE window.
// Title must have "vee" somewhere in it. Not case sensitive.
HWND GetMainWindow()
{
	UINT	i;
	HWND	h[50];
	char	title[1024];
	
	// Get all top-level thread windows.
	h[0] = (HWND)1;
	EnumThreadWindows(GetCurrentThreadId(), EnumWndProc, (LPARAM)h);
	
	// Find the one with "vee" in the title.
	for(i = 0; i < (UINT)h[0]; i++) {
		GetWindowText(h[i], title, 1023);
		if(str_contains_str(title, "vee")) return h[i];
	}
	
	// Not found will be reported as ERR_NO_VEE.
	return NULL;
}


// Subclass VEE window if found.
UINT SubclassWindow(BOOL bSubclass)
{
	HWND	h = NULL;
	char	title[] = "Subclass Failed";
	char	msg[80];

	// Init error status.
	guLastError = ERR_NO_ERROR;
	
	// Get main window handle (must have "vee" [not case sensitive] somewhere in title).
	h = GetMainWindow();

	// if subclassing...
	if(bSubclass) {
		
		// If found VEE window...
		if(h) {
			
			// Replace VEE's WndProc with our own.
			gWndProc = (WNDPROC)SetWindowLong(h, GWL_WNDPROC, (LONG)WndProc);
			if(!gWndProc) {
				
				// Fail if SetWindowLong fails, and report failure if notify is on.
				if(gbNotify) {
					wsprintf(msg, "Couldn't subclass window: %d", GetLastError());
					MessageBox(h, msg, title, MB_OK);
				}
				
				// In any case, report failure to caller.
				guLastError = ERR_NO_SUBCLASS;
				return guLastError;
			}
		
		// If VEE window was not found, report failure if notify is on.
		} else {
			if(gbNotify) {
				wsprintf(msg, "Couldn't find window with \"vee\" in title");
				MessageBox(NULL, msg, title, MB_OK);
			}
			
			// In any case, report failure to caller.
			guLastError = ERR_NO_VEE;
			return guLastError;
		}
	
	// If unsubclassing... (as in dll is being unloaded).
	} else {
		
		// Only try if the window is subclassed in the first place.
		if(gWndProc) {
			
			// Replace VEE's WndProc with the original VEE WndProc.
			if(!SetWindowLong(h, GWL_WNDPROC, (LONG)gWndProc)) {
				
				// Fail if SetWindowLong fails, and report failure if notify is on.
				if(gbNotify) {
					wsprintf(msg, "Couldn't unsubclass window: %d", GetLastError());
					MessageBox(h, msg, title, MB_OK);
				}
				
				// In any case, report failure to caller.
				guLastError = ERR_NO_UNSUBCLASS;
				return guLastError;
			}
		}
		
		// This can be reported as a failure: if Subclass(FALSE) is called
		// and the window isn't subclassed in the first place, it probably
		// only means that I did something stupid.
		guLastError = ERR_NO_SUBCLASS;
	}
	return guLastError;
}


// MAIN ENTRY POINT

// This is the function LoadLibrary calls.
BOOL WINAPI DllMain(HINSTANCE h, DWORD dwReason, LPVOID lpv)
{
	switch(dwReason) {
		
		
		// Initialize.
		case DLL_PROCESS_ATTACH:
			ghInst = h;
			gWndProc = NULL;
			guAclMs = guCtxMs = guLastError = guCancelCount = 0;
			gbMonitor = gbGotEnterMenuLoop = gbWasContextMenu = FALSE;
			gbNotify = TRUE;
			break;
		
		case DLL_THREAD_ATTACH:		break;
		case DLL_THREAD_DETACH:		break;
		
		// Clean up.
		case DLL_PROCESS_DETACH:
			if(gWndProc) {
				SubclassWindow(FALSE);
				gWndProc = NULL;
			}
			ghInst = NULL;
			break;
	}
	return TRUE;
}


// EXPORTS

VOID __stdcall StallGuard_SetKeyTimeoutMs(UINT uAclMs) {guAclMs = uAclMs;}

VOID __stdcall StallGuard_SetMouseTimeoutMs(UINT uCtxMs) {guCtxMs = uCtxMs;}

VOID __stdcall StallGuard_SetTimeoutsMs(UINT uAclMs, UINT uCtxMs) {guAclMs = uAclMs; guCtxMs = uCtxMs;}

VOID __stdcall StallGuard_SetFailNotify(BOOL bNotify) {gbNotify = bNotify;}

UINT __stdcall StallGuard_GetCancelCount(VOID) {return guCancelCount;}

VOID __stdcall StallGuard_ResetCancelCount(VOID) {guCancelCount = 0;}

UINT __stdcall StallGuard_GetLastError(VOID) {return guLastError;}

UINT __stdcall StallGuard_Monitor(BOOL bMonitor)
{
	guLastError = ERR_NO_ERROR;

	// If the window isn't subclassed yet, subclass it.
	if(!gWndProc) guLastError = SubclassWindow(TRUE);
	gbMonitor = bMonitor;
	return guLastError;
}

